package common.metrics.starter.filter;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import common.base.tools.type.NumberTools;
import common.config.tools.config.ConfigTools3;
import common.metrics.micrometer.registry.MetricsMeterRegistry;
import io.micrometer.core.instrument.DistributionSummary;
import io.micrometer.core.instrument.Timer;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;


/**
 * MetricsFilter: record API time consume
 */
public class MetricsFilter implements Filter {
    private MetricsMeterRegistry registry;

    public MetricsFilter(MetricsMeterRegistry registry) {
        this.registry = registry;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        Timer.Sample sample = Timer.start();
        try {
            chain.doFilter(request, response);
        } finally {
            HttpServletRequest httpRequest = (HttpServletRequest) request;
            HttpServletResponse httpResponse = (HttpServletResponse) response;

            String apiName = HttpTools.getRequestURI(httpRequest);

            int status = httpResponse.getStatus();

            Timer.Builder builder = Timer.builder("apiStat")
                    .tag("apiName", apiName)
                    .tag("status", String.valueOf(status));

            if (ConfigTools3.getBoolean("metrics.trace.api.session.enabled", false)) {
                //ip地址
                String clientIp = HttpTools.getRemoteHost(httpRequest);
                if (!Strings.isNullOrEmpty(clientIp)) {
                    builder.tag("clientIp", clientIp);
                }

                Map<String, String> requestMap = HttpTools.getRequestMap(httpResponse);
                requestMap.forEach((k, v) -> {
                    if (!Strings.isNullOrEmpty(k) && !Strings.isNullOrEmpty(v)) {
                        builder.tag(k, v);
                    }
                });
            }

            if (ConfigTools3.getBoolean("metrics.trace.api.sla.enable", true)) {
                builder.maximumExpectedValue(Duration.ofSeconds(15));
                builder.minimumExpectedValue(Duration.ofMillis(1));

                ArrayList<Duration> slaLevel = getSLALevel();
                if (!slaLevel.isEmpty()) {
                    Duration[] durations = new Duration[slaLevel.size()];
                    builder.serviceLevelObjectives(slaLevel.toArray(durations));
                }

                //百分比配置最多支持2个
                ArrayList<Double> slaPercentile = getSLAPercentiles();
                if (slaPercentile.size() == 1) {
                    builder.publishPercentiles(slaPercentile.get(0));
                }
                if (slaPercentile.size() == 2) {
                    builder.publishPercentiles(slaPercentile.get(0), slaPercentile.get(1));
                }

                builder.publishPercentileHistogram(ConfigTools3.getBoolean("metrics.trace.api.sla.publish.histogram", false));
            }

            long durationNs = sample.stop(builder.register(registry));
        }
    }

    @Override
    public void destroy() {

    }

    public ArrayList<Duration> getSLALevel() {
        ArrayList<Duration> slaDurations = Lists.newArrayList();

        List<String> slas = ConfigTools3.getAsList("metrics.trace.api.sla.threshold");
        if (slas.isEmpty()) {
            return slaDurations;
        }

        for (String sla : slas) {
            try {
                slaDurations.add(Duration.ofMillis(Long.valueOf(sla)));
            } catch (Exception e) {

            }
        }

        return slaDurations;
    }

    public ArrayList<Double> getSLAPercentiles() {
        ArrayList<Double> slaPercentiles = Lists.newArrayList();

        List<String> percentiles = ConfigTools3.getAsList("metrics.trace.api.sla.percentiles");
        if (percentiles.isEmpty()) {
            return slaPercentiles;
        }

        for (String percentile : percentiles) {
            try {
                slaPercentiles.add(Double.valueOf(percentile).doubleValue());
            } catch (Exception e) {

            }
        }

        return slaPercentiles;
    }
}

