package common.metrics.starter.filter;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import common.config.tools.config.ConfigTools3;
import common.metrics.micrometer.registry.MetricsMeterRegistry;
import io.micrometer.core.instrument.Timer;


import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;


/**
 * MetricsFilter: record API time consume
 */
public class MetricsFilter implements Filter {
    private MetricsMeterRegistry registry;

    public MetricsFilter(MetricsMeterRegistry registry) {
        this.registry = registry;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        //Exclude URL
        if (isExcludeUrls(request)) {
            chain.doFilter(request, response);
            return;
        }

        Timer.Sample sample = Timer.start();
        try {
            chain.doFilter(request, response);
        } finally {
            HttpServletRequest httpRequest = (HttpServletRequest) request;
            HttpServletResponse httpResponse = (HttpServletResponse) response;
            String apiName = HttpTools.getRequestURI(httpRequest);

            int status = httpResponse.getStatus();
            Timer.Builder builder = Timer.builder("apiStat")
                    .tag("apiName", apiName)
                    .tag("status", String.valueOf(status));

            //Region信息
            if (ConfigTools3.getBoolean("metrics.trace.api.region.enabled", true)) {
                String region = HttpTools.getClientRegion(httpRequest);
                if (!Strings.isNullOrEmpty(region)) {
                    builder.tag("xRegion", region);
                }
            }

            //sla
            if (ConfigTools3.getBoolean("metrics.trace.api.sla.enable", true)) {
                builder.maximumExpectedValue(Duration.ofSeconds(15));
                builder.minimumExpectedValue(Duration.ofMillis(1));

                ArrayList<Duration> slaLevel = getSLALevel();
                if (!slaLevel.isEmpty()) {
                    Duration[] durations = new Duration[slaLevel.size()];
                    builder.serviceLevelObjectives(slaLevel.toArray(durations));
                }

                //百分比配置最多支持2个
                ArrayList<Double> slaPercentile = getSLAPercentiles();
                if (slaPercentile.size() == 1) {
                    builder.publishPercentiles(slaPercentile.get(0));
                }
                if (slaPercentile.size() == 2) {
                    builder.publishPercentiles(slaPercentile.get(0), slaPercentile.get(1));
                }

                builder.publishPercentileHistogram(ConfigTools3.getBoolean("metrics.trace.api.sla.publish.histogram", false));
            }


            long durationNs = sample.stop(builder.register(registry));

            Map<String, String> requestMaps = getTags(httpRequest, httpResponse);

            //Assign user
            if (isAssignTrace(requestMaps)) {
                recordTime("apiTrace", apiName, durationNs, requestMaps);
            }

            //Slow response
            if (isSlowResponse(durationNs)) {
                recordTime("apiSlow", apiName, durationNs, requestMaps);
            }
        }
    }

    @Override
    public void destroy() {

    }

    /**
     * 获取SLA级别
     *
     * @return
     */
    public ArrayList<Duration> getSLALevel() {
        ArrayList<Duration> slaDurations = Lists.newArrayList();

        List<String> slas = ConfigTools3.getAsList("metrics.trace.api.sla.threshold");
        if (slas.isEmpty()) {
            return slaDurations;
        }

        for (String sla : slas) {
            try {
                slaDurations.add(Duration.ofMillis(Long.valueOf(sla)));
            } catch (Exception e) {

            }
        }

        return slaDurations;
    }

    /**
     * 获取SLA百分比
     *
     * @return
     */
    public ArrayList<Double> getSLAPercentiles() {
        ArrayList<Double> slaPercentiles = Lists.newArrayList();

        List<String> percentiles = ConfigTools3.getAsList("metrics.trace.api.sla.percentiles");
        if (percentiles.isEmpty()) {
            return slaPercentiles;
        }

        for (String percentile : percentiles) {
            try {
                slaPercentiles.add(Double.valueOf(percentile).doubleValue());
            } catch (Exception e) {

            }
        }

        return slaPercentiles;
    }

    /**
     * 是否是指定跟踪
     *
     * @param tagMap
     * @return
     */
    public boolean isAssignTrace(Map<String, String> tagMap) {

        List<String> traceMacs = ConfigTools3.getAsList("metrics.trace.api.macs");
        if (!traceMacs.isEmpty() && traceMacs.contains(Strings.nullToEmpty(tagMap.get("mac")))) {
            return true;
        }

        List<String> traceLoginIds = ConfigTools3.getAsList("metrics.trace.api.loginIds");
        if (!traceLoginIds.isEmpty() && traceLoginIds.contains(Strings.nullToEmpty(tagMap.get("loginId")))) {
            return true;
        }

        return false;
    }

    /**
     * 获取用户详情tag
     *
     * @param httpRequest
     * @param httpResponse
     * @return
     */
    public Map<String, String> getTags(HttpServletRequest httpRequest,
                                       HttpServletResponse httpResponse) {
        Map<String, String> tagMap = Maps.newHashMap();

        //ip地址
        String clientIp = HttpTools.getRemoteHost(httpRequest);
        if (!Strings.isNullOrEmpty(clientIp)) {
            tagMap.put("clientIp", clientIp);
        }

        tagMap = HttpTools.getRequestMap(httpResponse);
        if (!tagMap.containsKey("mac")) {
            String did = HttpTools.getArgs(httpRequest, "mac");
            if (Strings.isNullOrEmpty(did)) {
                did = HttpTools.getArgs(httpRequest, "did");
            }

            if (!Strings.isNullOrEmpty(did)) {
                tagMap.put("mac", did);
            }
        }

        if (!tagMap.containsKey("loginId")) {
            String loginId = HttpTools.getArgs(httpRequest, "loginId");
            if (!Strings.isNullOrEmpty(loginId)) {
                tagMap.putIfAbsent("loginId", loginId);
            }
        }

        return tagMap;
    }

    /**
     * 是否是慢速响应
     *
     * @param durationNs
     * @return
     */
    public boolean isSlowResponse(long durationNs) {
        long slowThreshold = ConfigTools3.getLong("metrics.trace.api.slow.threshold", 3000L);
        if (durationNs >= Duration.ofMillis(slowThreshold).toNanos()) {
            return true;
        }
        return false;
    }

    /**
     * 记录时长
     *
     * @param timerName:timer名称
     * @param apiName：api       名称
     * @param durationNs:时长
     * @param tags:标签
     */
    private void recordTime(String timerName, String apiName, long durationNs, Map<String, String> tags) {
        Timer.Builder traceTimer = Timer.builder(timerName);
        traceTimer.tag("apiName", apiName);
        tags.forEach((k, v) -> {
            if (!Strings.isNullOrEmpty(k) && !Strings.isNullOrEmpty(v)) {
                traceTimer.tag(k, v);
            }
        });
        traceTimer.register(registry).record(Duration.ofNanos(durationNs));
    }

    private boolean isExcludeUrls(ServletRequest request) {
        String url = ((HttpServletRequest) request).getRequestURI();
        if (Strings.isNullOrEmpty(url)) {
            return false;
        }
        url = url.toLowerCase();
        //Ignore ping
        if (url.startsWith("/api/ping")) {
            return true;
        }

        List<String> excludeUrlList = ConfigTools3.getAsList("metrics.url.exclude");
        if (excludeUrlList.isEmpty()) {
            return false;
        }

        if (excludeUrlList.contains("/*")) {
            return true;
        }

        for (String urlPath : excludeUrlList) {
            if (url.startsWith(urlPath.toLowerCase())) {
                return true;
            }
        }

        return false;
    }
}

