package com.valor.web.springboot.starter.goose.eus.token.interceptor;

import com.google.common.base.Strings;
import com.google.gson.Gson;
import com.valor.web.springboot.starter.goose.eus.token.service.ETokenStatus;
import com.valor.web.springboot.starter.goose.eus.token.service.GooseEUSTokenService;
import comm.base.tools.api.model.ApiBaseResponse;
import common.config.tools.config.ConfigTools3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;

public class GooseEUSTokenInterceptor extends HandlerInterceptorAdapter {
    private Logger logger = LoggerFactory.getLogger(GooseEUSTokenInterceptor.class);

    public static final int ERR_TOKEN_EXPIRED = 7001001;
    public static final int ERR_TOKEN_INVALID = 7001002;

    private GooseEUSTokenService apiTokenService = new GooseEUSTokenService();
    private Gson gson = new Gson();

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        if (isRequireVerifyToken(request)){
            //do token verify
            ETokenStatus status = verifyToken(request, response);
            if (!status.isSuccess()) {
                //Token 校验失败,返回错误新
                setErrorResponse(response, status);
                return false;
            }
        }

        return true;
    }


    public String getToken(HttpServletRequest request) {
        return getParameter(request, "token");
    }

    public String getMac(HttpServletRequest request) {
        return getParameter(request, "mac");
    }

    public String getLoginId(HttpServletRequest request) {
        return getParameter(request, "loginId");
    }

    /**
     * 是否需要校验token
     *
     * @param request
     * @return
     */
    public Boolean isRequireVerifyToken(HttpServletRequest request) {
        String url = request.getRequestURI();

        List<String> excludeUrls = ConfigTools3.getAsList("token.verify.goose.eus.url.exclude");
        for (String e : excludeUrls) {
            if (url.startsWith(e)) {
                return false;
            }
        }

        return true;
    }

    public ETokenStatus verifyToken(HttpServletRequest request, HttpServletResponse response) {
        String token = getToken(request);
        String mac = getMac(request);
        String loginId = getLoginId(request);

        return apiTokenService.checkToken(token, mac, loginId);
    }

    private void setErrorResponse(HttpServletResponse response, ETokenStatus status) {
        if (status.isSuccess()) {
            return;
        }

        ApiBaseResponse apiResponse = new ApiBaseResponse();
        if (status == ETokenStatus.EXPIRE) {
            apiResponse.setErrCode(ERR_TOKEN_EXPIRED);
            apiResponse.setMessage(String.format("Invalid Token [%s]", status.toString()));
        } else {
            apiResponse.setErrCode(ERR_TOKEN_INVALID);
            apiResponse.setMessage(status.toString());
        }

        try {

            if (!response.isCommitted()) {
                response.resetBuffer();
            }
            response.setContentType("application/json");
            String payload = gson.toJson(apiResponse);
            OutputStream out = response.getOutputStream();
            out.write(payload.getBytes(StandardCharsets.UTF_8));

        } catch (Exception e) {
            logger.info("Token verify failed!", e);
            response.setStatus(HttpServletResponse.SC_SERVICE_UNAVAILABLE);
        }
    }

    private String getParameter(HttpServletRequest request, String key) {
        /**
         * 获取参数值,获取顺序:parameter->header
         */
        String v = "";
        v = request.getHeader(key);
        if (!Strings.isNullOrEmpty(v)) {
            return v;
        }

        v = request.getParameter(key);
        if (!Strings.isNullOrEmpty(key)) {
            return v;
        }

        return "";
    }
}
