package com.valor.web.springboot.starter.goose.portal.token.service;


import com.google.common.base.Strings;
import com.valor.web.springboot.starter.goose.portal.token.model.UserDetailRequest;
import com.valor.web.springboot.starter.goose.portal.token.model.UserDetailResponse;
import com.valor.web.springboot.starter.goose.portal.token.model.GooseWebApiResponse;
import com.valor.web.springboot.starter.goose.portal.token.service.config.GooseApiConfig;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

public class GoosePortalApi {
    private final Logger logger = LoggerFactory.getLogger(GoosePortalApi.class);
    public static final int ERR_TOKEN_EXPIRE = 10200002;

    private Retrofit retrofit;
    private String hosts;
    private int connectTimeout = 10000;
    private int readTimeout = 10000;

    private IGoosePortalApi gooseWebApi;

    public GoosePortalApi() {
        this(GooseApiConfig.goosePortalApiHosts(),
                GooseApiConfig.goosePortalApiConnectTimeout(),
                GooseApiConfig.goosePortalApiReadTimeout());
    }

    public GoosePortalApi(String hosts) {
        this(hosts, 10_000, 10_000);
    }

    public GoosePortalApi(String hosts, int connectTimeout, int readTimeout) {
        this.hosts = hosts;
        this.connectTimeout = connectTimeout;
        this.readTimeout = readTimeout;

        if (Strings.isNullOrEmpty(hosts)) {
            throw new IllegalArgumentException("Goose portal api host not config");
        }

        OkHttpClient okHttpClient = new OkHttpClient.Builder()
                .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
                .readTimeout(readTimeout, TimeUnit.MILLISECONDS)
                .build();

        retrofit = new Retrofit.Builder()
                .baseUrl(hosts)
                .client(okHttpClient)
                .addConverterFactory(GsonConverterFactory.create())
                .build();

        gooseWebApi = retrofit.create(IGoosePortalApi.class);
        logger.info("GoosePortalApi host[{}] timeout[{},{}]", hosts, connectTimeout, readTimeout);
    }

    public GooseWebApiResponse<UserDetailResponse> getUserDetail(UserDetailRequest request) {
        try {
            Response<GooseWebApiResponse<UserDetailResponse>> response = gooseWebApi.getUserDetailV1(request.getOperatorId(), request.getToken()).execute();
            if (response.isSuccessful()) {
                GooseWebApiResponse<UserDetailResponse> apiResponse = response.body();
                if (apiResponse.getErrCode() != 0) {
                    logger.error("Service Failed,status:{} response:{}", apiResponse.getErrCode(), apiResponse.getMessage());
                }
                return apiResponse;
            } else {
                logger.error("Http Failed,status:{} response:{}", response.code(), response.errorBody());
                return new GooseWebApiResponse(response.code(), String.format("Http status is not 2xx,code[%s]", response.code()));
            }
        } catch (IOException e) {
            logger.error("Http exception", e);
            return new GooseWebApiResponse(500, String.format("Http exception[%s]", e.getMessage()));
        }
    }
}
