package com.valor.web.springboot.starter.goose.portal.token.service;


import com.google.common.base.Strings;
import com.valor.web.springboot.starter.goose.portal.token.service.config.CacheConfig;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.Config;
import org.redisson.config.TransportMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RedissonCacheClient {
    private final Logger logger = LoggerFactory.getLogger("RedisClient");
    private RedissonClient client;
    private static RedissonCacheClient instance = new RedissonCacheClient();

    private RedissonCacheClient() {
    }

    public static RedissonCacheClient getInstance() {
        return instance;
    }

    public RedissonClient getRedissionClient() {
        if (client != null && !client.isShutdown()) {
            return client;
        }

        synchronized (this) {
            client = createRedissionClient();
        }

        return client;
    }

    private Config createConfig() {
        Config config = new Config();
        config.setTransportMode(getTransportMode(CacheConfig.transportMode()));
        logger.info("Cache:TransportMode={}", config.getTransportMode().name());
        ClusterServersConfig clusterConfig = config.useClusterServers();
        if (CacheConfig.nodeAddresses().isEmpty()) {
            throw new IllegalArgumentException("Redis node address not Config");
        }

        CacheConfig.nodeAddresses().forEach(e -> clusterConfig.addNodeAddress(e));
        if (!Strings.isNullOrEmpty(CacheConfig.password())) {
            clusterConfig.setPassword(CacheConfig.password());
        }

        logger.info("Cache:Nodes={}", clusterConfig.getNodeAddresses());
        return config;
    }

    private TransportMode getTransportMode(String transportMode) {
        if (Strings.isNullOrEmpty(transportMode)) {
            boolean isUnix = Strings.nullToEmpty(System.getProperties().getProperty("os.name")).toLowerCase().contains("linux");
            if (isUnix) {
                return TransportMode.EPOLL;
            }
            return TransportMode.NIO;
        }

        switch (transportMode) {
            case "EPOLL":
                return TransportMode.EPOLL;
            case "KQUEUE":
                return TransportMode.KQUEUE;

            default:
                return TransportMode.NIO;
        }
    }

    private RedissonClient createRedissionClient() {
        return Redisson.create(createConfig());
    }
}
