package com.valor.web.springboot.starter.i18n.filter;

import com.google.common.base.Strings;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.valor.web.springboot.starter.i18n.utils.I18nMsgBuilder;
import com.valor.web.utils.wrapper.WebResponseWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;


/**
 * I18N Filter
 */
public class I18nFilter extends OncePerRequestFilter {
    private Logger logger = LoggerFactory.getLogger("I18nFilter");

    private static JsonParser gsonParser = new JsonParser();
    public static final String NAME_ERR_CODE = "errCode";
    public static final String NAME_MESSAGE = "message";


    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        WebResponseWrapper wrapper = new WebResponseWrapper (response);
        try{
            filterChain.doFilter(request, wrapper);

            byte[] contentByte = wrapper.getContentAsByteArray();
            if (contentByte.length > 0) {
                String content = new String(contentByte);
                i18nContent(content, request, wrapper);
            }
        }finally {
            wrapper.copyBodyToResponse();
        }
    }


    private void i18nContent(String content, HttpServletRequest request, HttpServletResponse response) {
        try {
            JsonElement element = gsonParser.parse(content);
            if (element.isJsonNull()) {
                return;
            }

            if (!element.isJsonObject()) {
                return;
            }

            JsonObject jsonObject = element.getAsJsonObject();
            if (!jsonObject.has(NAME_ERR_CODE)) {
                return;
            }

            int code = jsonObject.get(NAME_ERR_CODE).getAsInt();
            if (code == 0) {
                return;
            }

            //获取设置的错误消息作为默认
            String defaultMsg = "";
            if (jsonObject.has(NAME_MESSAGE)){
                defaultMsg = jsonObject.get(NAME_ERR_CODE).getAsString();
            }

            String i18nMsg = getI18nMsg(code, request, defaultMsg);
            if (!Strings.isNullOrEmpty(i18nMsg)) {
                jsonObject.addProperty(NAME_MESSAGE, i18nMsg);
            }

            String newContent = jsonObject.toString();
            if (!response.isCommitted()) {
                response.resetBuffer();
            }
            response.getOutputStream().write(newContent.getBytes(StandardCharsets.UTF_8));
        } catch (Exception e) {

        }
    }


    public String getLang(HttpServletRequest request) {
        String lang = "en";
        lang = getParameter(request, "lang");
        if (!Strings.isNullOrEmpty(lang)) {
            return lang;
        }

        lang = getParameter(request, "language");
        if (!Strings.isNullOrEmpty(lang)) {
            return lang;
        }

        return "en";
    }

    public String getUid(HttpServletRequest request) {
        String uid = getParameter(request, "loginId");
        if (!Strings.isNullOrEmpty(uid)) {
            return uid;
        }

        uid = getParameter(request, "mac");
        if (!Strings.isNullOrEmpty(uid)) {
            return uid;
        }

        uid = getParameter(request, "accountId");
        if (!Strings.isNullOrEmpty(uid)) {
            return uid;
        }

        return "";
    }

    public String getI18nMsg(int code, HttpServletRequest request, String defaultMsg) {
        String lang = getLang(request);
        String uid = getUid(request);
        return I18nMsgBuilder.buildMsg(lang, code, uid, defaultMsg);
    }

    private String getParameter(HttpServletRequest request, String key) {
        HttpServletRequest httpRequest = request;
        /**
         * 获取参数值,获取顺序:parameter->header
         */
        String v = "";
        v = httpRequest.getHeader(key);
        if (!Strings.isNullOrEmpty(v)) {
            return v;
        }

        v = httpRequest.getParameter(key);
        if (!Strings.isNullOrEmpty(key)) {
            return v;
        }

        return "";
    }
}
