package com.valor.web.utils.http;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import common.base.tools.type.NumberTools;
import common.config.tools.config.ConfigTools3;

import javax.servlet.http.HttpServletRequest;
import java.util.List;

public class HttpTools {

    private static ImmutableList<String> ipHeaderNameList = ImmutableList.<String>builder()
        .add("x-forwarded-for")
        .add("CF-Connecting-IP")
        .add("Proxy-Client-IP")
        .add("WL-Proxy-Client-IP")

        .add("X-Forwarded-For")
        .add("HTTP_X_REAL_IP")
        .add("HTTP_CLIENT_IP")
        .add("HTTP_X_FORWARDED_FOR")

        .build();

    public static List<String> getIpHeaderNameList(){
        List<String> ipHeaderNames = Lists.newLinkedList(ipHeaderNameList);
        List<String> ipHeaderFromCfg = ConfigTools3.getAsList("ip.addr.header.name");
        if (!ipHeaderFromCfg.isEmpty()){
            ipHeaderNames.addAll(ipHeaderFromCfg);
        }

        return ipHeaderNames;
    }

    private static boolean isInvalidClientIp(String clientIp) {
        if (Strings.isNullOrEmpty(clientIp)) {
            return false;
        }

        if ("unknown".equalsIgnoreCase(clientIp)) {
            return false;
        }

        return true;
    }

    public static String getRemoteHost(HttpServletRequest request) {

        /*
           Get real client ip,support CloudFlare,Aliyun,AWS proxy
         */
        String remoteHost = null;
        for (String headerName : ipHeaderNameList) {
            remoteHost = request.getHeader(headerName);
            if (isInvalidClientIp(remoteHost)) {
                break;
            }
        }

        //CAN NOT get from header,get from request
        if (!isInvalidClientIp(remoteHost)) {
            remoteHost = request.getRemoteAddr();
        }

        //对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if (!Strings.isNullOrEmpty(remoteHost) && remoteHost.length() > 15) { //"***.***.***.***".length() = 15
            if (remoteHost.indexOf(",") > 0) {
                remoteHost = remoteHost.substring(0, remoteHost.indexOf(","));
            }
        }

        return Strings.nullToEmpty(remoteHost);
    }


    public static String getHttpParameter(HttpServletRequest request, String key) {
        if (request == null) {
            return "";
        }

        if (Strings.isNullOrEmpty(key)) {
            return "";
        }

        return request.getParameter(key);
    }

    public static Long getLongParameter(HttpServletRequest request, String key) {
        String v = getHttpParameter(request, key);
        if (Strings.isNullOrEmpty(v)) {
            return null;
        }

        return Long.valueOf(v);
    }

    public static Long getLongAttr(HttpServletRequest request, String key) {
        return getLongAttr(request, key, null);
    }

    public static Long getLongAttr(HttpServletRequest request, String key, Long defaultValue) {
        Object v = request.getAttribute(key);
        if (v == null) {
            return defaultValue;
        }

        if (v instanceof Long) {
            return (Long) v;
        } else if (v instanceof String) {
            if (Strings.isNullOrEmpty((String) v)) {
                return defaultValue;
            }

            return Long.valueOf((String) v);
        } else {
            return defaultValue;
        }
    }

    public static int getIntAttr(HttpServletRequest request, String key, int defaultValue) {
        Object v = request.getAttribute(key);
        if (v == null) {
            return defaultValue;
        }

        if (v instanceof Integer) {
            return (int) v;
        }

        if (v instanceof String) {
            return NumberTools.str2int((String) v);
        }

        return defaultValue;
    }


    public static String getAttr(HttpServletRequest request, String key, String defaultValue) {
        Object v = request.getAttribute(key);
        if (v == null) {
            return defaultValue;
        }

        return (String) v;
    }

    public static void setAttrIfAbsent(HttpServletRequest request, String key, Object value) {
        Object v = request.getAttribute(key);
        if (v == null) {
            request.setAttribute(key, value);
        }
    }

    public static void setAttr(HttpServletRequest request, String key, Object value) {
        request.setAttribute(key, value);
    }

    public static String getApiPath(HttpServletRequest request) {
        return request.getRequestURI();
    }

    public static String getRequestURI(HttpServletRequest request) {
        String apiName = request.getRequestURI();
        apiName = apiName.substring(1).replace("/", "_");
        return apiName;
    }

    public static String getClientRegion(HttpServletRequest request) {
        return Strings.nullToEmpty(request.getHeader("CF-IPCountry"));
    }

}
