package com.valor.gray.feignextension.header;

import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import feign.RequestInterceptor;
import feign.RequestTemplate;

/**
 * feign的header拦截器，用于header透传 注意：此透传方式只适用于对feign的同步调用，异步的话会获取不到header
 * 
 * @author lihe
 * @date 2021/08/20
 */
@Component
public class FeignHeaderInterceptor implements RequestInterceptor {
    private static final Logger logger = LogManager.getLogger(FeignHeaderInterceptor.class);
    /**
     * 要进行透传所有header的名称
     */
    @Value("#{'${header.names}'.split(',')}")
    private List<String> headerNameList;

    @Override
    public void apply(RequestTemplate template) {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            // 获取不到原请求，可能是对feign的调用为异步导致
            logger.error("[feign extension] header pass failed: RequestContextHolder.getRequestAttributes() is null");
            return;
        }
        HttpServletRequest oldRequest = ((ServletRequestAttributes)requestAttributes).getRequest();
        passHeader(oldRequest, template);
    }

    /**
     * header透传
     * 
     * @param oldRequest
     * @param template
     */
    private void passHeader(HttpServletRequest oldRequest, RequestTemplate template) {
        for (String headerName : headerNameList) {
            String headerValue = oldRequest.getHeader(headerName);
            if (StringUtils.isEmpty(headerValue)) {
                logger.debug(String.format(
                    "[feign extension] header ignore: target header value is empty, headerName = %s", headerName));
            } else {
                logger.debug(String.format(
                    "[feign extension] header ignore: target header value is empty, headerName = %s", headerName));
                template.header(headerName, headerValue);
            }
        }
    }
}
