 package com.valor.gray.feignextension.ribbon;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import com.netflix.loadbalancer.ServerListFilter;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;

/**
 * 灰度过滤器
 * @author lihe
 * @date 2021/08/20
 */
public class  GrayFilter implements ServerListFilter<DiscoveryEnabledServer>{
    private static final Logger logger = LogManager.getLogger(GrayFilter.class);

    @Override
    public List<DiscoveryEnabledServer> getFilteredListOfServers(List<DiscoveryEnabledServer> servers) {
         // 过滤逻辑：如果有灰度版本服务实例，则返回灰度的，没有的话返回所有
         String targetVersion = this.getTargetVersion();
         if (StringUtils.isEmpty(targetVersion)) {
             logger.debug("[feign extension] gray server filter: return all servers, because targetVersion is empty");
             return servers;
         }
         List<DiscoveryEnabledServer> targetVersionServers = new ArrayList<>();
         for (DiscoveryEnabledServer server : servers) {
             String version = server.getInstanceInfo().getMetadata().get("version");
             if (StringUtils.isEmpty(version)) {
                 logger.debug("[feign extension] gray server filter: server({}) ignore, because server version is empty", server.getInstanceInfo().getAppName());
                 continue;
             }
             if (version.equals(targetVersion)) {
                 targetVersionServers.add(server);
             }
         }
         if (CollectionUtils.isEmpty(targetVersionServers)) {
             logger.debug("[feign extension] gray server filter: return normal server");
             return servers;
         } else {
             // 有灰度实例
             logger.debug("[feign extension] gray server filter: return gray server");
             return targetVersionServers;
         }
    }
    
    /**
     * 从请求头中获取目标版本号
     * @return
     */
    private String getTargetVersion () {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            // 获取不到原请求，可能是对feign的调用为异步导致
            logger.error("[feign extension] gray server filter failed: RequestContextHolder.getRequestAttributes() is null");
            return null;
        }
        HttpServletRequest oldRequest = ((ServletRequestAttributes)requestAttributes).getRequest();
        return oldRequest.getHeader("targetVersion");
    }
}
