package com.mfc.account.stub;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import common.base.tools.exception.ApiException;
import java.io.IOException;
import java.util.Map;
import org.jsoup.Connection;
import org.jsoup.Jsoup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by Frank.Huang on 2017/1/19.
 */
public class HttpInvokeTools {

    private HttpInvokeTools() {
        throw new IllegalStateException("Utility class");
    }

    private static final Logger logger = LoggerFactory.getLogger(HttpInvokeTools.class);

    public static String getInvoke(String url, int timeout) throws ApiException {
        try {
            Connection.Response httpResponse = Jsoup.connect(url)
                    .timeout(timeout)
                    .ignoreContentType(true)
                    .ignoreHttpErrors(true)
                    .method(Connection.Method.GET)
                    .execute();
            return getResponseBodyString(httpResponse);
        } catch (IOException e) {
            logger.error("Url[{}] Exception:{}", url, e);
            throw new ApiException(AccountStubRetCode.RET_ACCOUNT,
                AccountStubRetCode.ERR_STUB_JSOUP_EXCEPTION, e.getMessage());
        }
    }

    public static String postInvoke(String url, int timeout, Map<String, Object> dataMap) throws ApiException {
        try {
            Connection httpConnection = Jsoup.connect(url)
                    .timeout(timeout)
                    .ignoreContentType(true)
                    .ignoreHttpErrors(true)
                    .method(Connection.Method.POST);
            httpConnection.header("Content-Type", "application/json;charset=UTF-8");
            httpConnection.requestBody(toJson(dataMap));
            Connection.Response httpResponse = httpConnection.execute();
            return getResponseBodyString(httpResponse);
        } catch (IOException e) {
            logger.error("Url[{}] Exception:{}", url, e);
            throw new ApiException(AccountStubRetCode.RET_ACCOUNT,
                AccountStubRetCode.ERR_STUB_JSOUP_EXCEPTION, e.getMessage());
        }
    }

    public static <T> T httpResponse2Object(String jsonResponse, TypeReference reference) throws ApiException {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        try {
            return objectMapper.readValue(jsonResponse, reference);
        } catch (IOException e) {
            logger.error("[USER3 API STUB] Exception:{}", e);
            throw new ApiException(AccountStubRetCode.RET_ACCOUNT,
                AccountStubRetCode.ERR_STUB_JSON_EXCEPTION, e.getMessage());
        }
    }

    public static String toJson(Object objs) throws ApiException {
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            return objectMapper.writeValueAsString(objs);
        } catch (IOException e) {
            logger.error("[USER3 API STUB] Exception:{}", e);
            throw new ApiException(AccountStubRetCode.RET_ACCOUNT,
                AccountStubRetCode.ERR_STUB_JSON_EXCEPTION, e.getMessage());
        }
    }

    public static String buildQueryUrl(String host, String path, Map<String, Object> dataMap) {
        String queryString = Joiner.on("&").useForNull("").withKeyValueSeparator("=").join(dataMap);
        return host + path + "?" + queryString;
    }

    private static String getResponseBodyString(Connection.Response httpResponse) throws ApiException {
        if (httpResponse.statusCode() == 200 || httpResponse.statusCode() == 555) {
            String jsonResponse = httpResponse.body();
            if (Strings.isNullOrEmpty(jsonResponse)) {
                throw new ApiException(AccountStubRetCode.RET_ACCOUNT,
                    AccountStubRetCode.ERR_STUB_HTTP_RESPONSE_EMPTY, "Http response is empty.");
            }
            return jsonResponse;
        } else {
            throw new ApiException(AccountStubRetCode.RET_ACCOUNT,
                AccountStubRetCode.ERR_STUB_HTTP_RESPONSE_NOT_OK, "Http status code:" + httpResponse.statusCode());
        }
    }
}
