package com.common.web.aop.processor;

import com.common.web.aop.annotation.WebApiCallV1;
import com.valor.mfc.vms.api.model.common.response.ResponseStatus;
import com.valor.mfc.vms.api.model.constant.response.HttpCode2;
import com.valor.mfc.vms.common.tools.http.HttpTools;
import com.valor.mfc.vms.common.web.request.BaseRequestArgs;
import com.valor.mfc.vms.common.web.request.IRequestBaseArgs;
import common.base.tools.exception.ApiException;
import common.base.tools.statistics.CountStatisticsTools2;
import common.config.tools.config.ConfigTools3;
import common.web.tools.http.model.response.WebApiBaseResponse;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;


/**
 * Created by Frank.Huang on 2016/6/20.
 */
@Service
@Aspect
@Order(1)
public class WebApiCallInterceptorV1 {
    private static final Logger logger = LoggerFactory.getLogger(WebApiCallInterceptorV1.class);


    @Pointcut(value = "@annotation(com.common.web.aop.annotation.WebApiCallV1)")
    public void webApiCall() {
    }

    @Around(value = "com.common.web.aop.processor.WebApiCallInterceptorV1.webApiCall()&&@annotation(wac)")
    public Object apiRun(ProceedingJoinPoint pjp, WebApiCallV1 wac) throws ServletException, IOException, ApiException {
        String apiName = getApiName(pjp);
        String apiPath = getApiPath(pjp);


        CountStatisticsTools2.addCount(apiName, "REQ", 1);
        HttpServletRequest request = getRequest(pjp);
        HttpServletResponse response = getResponse(pjp);
        String host = HttpTools.getRemoteHost(request);



        //更新参数
        BaseRequestArgs baseArgs = getArgs(pjp);

        System.out.println(">>>> BaseRequestArgs: "+ obj2String(baseArgs));

        if (baseArgs != null) {
            baseArgs.setArgsFromParameter(request);
        }

        logRequest(wac, host, apiName, baseArgs);
        try {
            if (getMinVersion(wac, apiName) > 0) {
                Long appVer = HttpTools.getLongParameter(request, "appVer");
                if (appVer == null || appVer < getMinVersion(wac, apiName)) {
                    throw new ApiException(HttpCode2.RET_UNSUPPORTED_VERSION,
                        HttpCode2.ERR_UNSUPPORTED_VER, apiPath, "Unsupported version");
                }
            }

            Object retVal = pjp.proceed();

            //计算签名
            HttpTools.updateSrvInfo(request, retVal);

            boolean isSuccess = isSuccess(retVal);
            if (isSuccess) {
                CountStatisticsTools2.addCount(apiName, "RSP", 1);
                //设置cache control 时间
                if (getCacheControl(wac, apiName) > 0) {
                    getResponse(pjp).setHeader("Cache-Control", "max-age=" + wac.cacheControl());
                }
            } else {
                //如果失败，修改http code为555
                response.setStatus(HttpCode2.SERVER_ERROR);
                CountStatisticsTools2.addCount(apiName, "ERR", 1);
            }

            if (isSuccess) {
                logResponse(wac, host, apiName, retVal, baseArgs);
            } else {
                logResponse(false, host, apiName, retVal, baseArgs);
            }

            return retVal;
        } catch (Throwable throwable) {
            if (throwable instanceof ApiException) {
                throw (ApiException) throwable;
            }
            logger.error(String.format("API:[%s]",apiName) , throwable);
            throw new ApiException(HttpCode2.RET_SYS_EXCEPTION, HttpCode2.ERR_SYS_EXCEPTION, "Internal error");
        }
    }

    public HttpServletRequest getRequest(JoinPoint jp) {
        return (HttpServletRequest) jp.getArgs()[0];
    }

    public HttpServletResponse getResponse(JoinPoint jp) {
        return (HttpServletResponse) jp.getArgs()[1];
    }

    public BaseRequestArgs getArgs(JoinPoint jp) {
        Object arg = null;
        if (jp.getArgs().length >= 3) {
            arg = jp.getArgs()[2];
            if (arg instanceof BaseRequestArgs) {
                return (BaseRequestArgs) arg;
            }
        }
        return null;
    }

    public String getApiName(JoinPoint jp) {
        String apiName = getRequest(jp).getRequestURI();
        apiName = apiName.substring(1).replace("/", "-");
        return apiName;
    }

    public String getApiPath(JoinPoint jp) {
        return getRequest(jp).getRequestURI();
    }

    private void logRequest(WebApiCallV1 webApiCall,
                            String host,
                            String apiName,
                            IRequestBaseArgs args) {
        if (getLogLevel(webApiCall, apiName, "request") == 0) {
            return;
        }

        logReq(host, apiName, obj2String(args));
    }

    private void logResponse(WebApiCallV1 webApiCall,
                             String host,
                             String apiName,
                             Object result,
                             IRequestBaseArgs iArgs) {
        if (getLogLevel(webApiCall, apiName, "response") == 0) {
            return;
        }

        logResponse(true, host, apiName, result, iArgs);
    }

    private void logResponse(boolean isSuccess, String host, String apiName, Object result, IRequestBaseArgs iArgs) {
        if (isSuccess) {
            logRsp(host, apiName, obj2String(iArgs), obj2String(result));
        } else {
            logErrRsp(host, apiName, obj2String(iArgs), obj2String(result));
        }
    }

    private boolean isSuccess(Object result) {
        if (result instanceof ResponseStatus) {
            return (((ResponseStatus) result).getRetCode() == 0);
        } else if (result instanceof WebApiBaseResponse) {
            return (((WebApiBaseResponse) result).getRetCode() == 0);
        }

        return false;
    }


    private int getMinVersion(WebApiCallV1 webApiCall, String apiName) {
        final String minVerKey = "webapi.version.min";
        final String apiVerKey = String.format("%s.%s", minVerKey, apiName);

        int minVersion = ConfigTools3.getInt(apiVerKey, ConfigTools3.getInt(minVerKey, 0));
        if (minVersion == 0) {
            minVersion = webApiCall.minVersion();
        }

        return minVersion;
    }

    private int getLogLevel(WebApiCallV1 webApiCall, String apiName, String msgType) {
        final String defaultKey = "webapi.log.level";
        final String apiMsgKey = String.format("%s.%s.%s", defaultKey, msgType, apiName);
        final String apiKey = String.format("%s.%s", defaultKey, apiName);

        int value = ConfigTools3.getInt(apiMsgKey, 0);
        if (value != 0) {
            return value;
        }

        value = ConfigTools3.getInt(apiKey, 0);
        if (value != 0) {
            return value;
        }

        value = ConfigTools3.getInt(defaultKey, 0);
        if (value != 0) {
            return value;
        }

        value = webApiCall.logLevel();
        return value;
    }

    private int getCacheControl(WebApiCallV1 webApiCall, String apiName) {
        final String defaultKey = "webapi.cache.control";
        final String apiKey = String.format("%s.%s", defaultKey, apiName);

        int value = ConfigTools3.getInt(apiKey, 0);
        if (value != 0) {
            return value;
        }

        value = ConfigTools3.getInt(defaultKey, 0);
        if (value != 0) {
            return value;
        }

        return webApiCall.cacheControl();
    }

    private String obj2String(Object obj) {
        if (obj == null) {
            return "-";
        } else {
            return obj.toString();
        }
    }

    private void logReq(String host,
                        String apiName,
                        String request) {
        logger.info("[REQ] [HOST]:[{}] [API]:[{}]req[{}]",
            host, apiName, request);
    }

    private void logRsp(String host,
                        String apiName,
                        String request,
                        String response) {
        logger.info("[RSP] [HOST]:[{}] [API]:[{}] Resp[{}] req[{}]",
            host, apiName, response, request);
    }

    private void logErrRsp(String host,
                           String apiName,
                           String request,
                           String response) {
        logger.error("[RSP] [HOST]:[{}] [API]:[{}] Resp[{}] req[{}]",
            host, apiName, response, request);
    }

}
