package com.common.web.filter.sso;


import com.google.common.base.Strings;
import com.valor.mfc.vms.api.common.encrypt.RSABase64Utils;
import com.valor.mfc.vms.api.model.constant.response.HttpCode2;
import com.valor.mfc.vms.common.tools.http.ASKeyConfig;
import com.valor.mfc.vms.common.tools.http.HttpConstant;
import com.valor.mfc.vms.common.tools.http.HttpParameterTools;
import com.valor.mfc.vms.common.tools.http.HttpTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;

import java.util.Objects;
import java.util.Set;

/**
 * Created by Frank.Huang on 2016/6/25.
 */
public class SSOTokenFilter extends AbstractSSOFilter {
    private static final Logger logger = LoggerFactory.getLogger(SSOTokenFilter.class);

    @Override
    public String getToken(HttpServletRequest httpRequest) {
        return HttpParameterTools.getParameter(httpRequest, HttpConstant.HTTP_ARG_TOKEN);
    }

    @Override
    public byte[] decryptToken(String token) {
        try {
            return RSABase64Utils.decryptByPublicKey(token, ASKeyConfig.getAsKey());
        } catch (Exception e) {
            logger.error("{}", e);
            return new byte[0];
        }
    }

    @Override
    public int verifyToken(HttpServletRequest httpRequest, Object object) {
        String host = HttpTools.getRemoteHost(httpRequest);
        if (object instanceof TokenV2) {
            TokenV2 token = (TokenV2) object;
            logger.info("=======TokenV2-[{}]",token.toSerializeString());
            long curMS = System.currentTimeMillis();
            if (curMS >= token.getExpireTimestamp()) {
                logger.error("==========HOST:[{}] MAC:[{}] Token expire[{}-{}-{}]", host, token.getDid(),
                    curMS, token.getExpireTimestamp(), token.getExpireTimestamp());
                return HttpCode2.ERR_AUTH_EXPIRED_TOKEN;
            }

            Long uid = HttpParameterTools.getLongParameter(httpRequest, HttpConstant.HTTP_ARG_UID);
            if (uid != null && uid > 0 && uid.longValue() != token.getDid2()) {
                logger.error("==========HOST:[{}] QueryString[{}]", host,httpRequest.getQueryString());
                logger.error("==========HOST:[{}] uid:[{}] INVALID UID[{}]", host, uid, token.getDid2());
                return HttpCode2.ERR_AUTH_UID_CONFLICT;
            }

            Long accountId = HttpParameterTools.getLongParameter(httpRequest, HttpConstant.HTTP_ARG_ACCOUNT_ID);
            if (accountId != null && accountId > 0 && accountId.longValue() != token.getAccountId()) {
                logger.error("==========HOST:[{}] QueryString[{}]", host,httpRequest.getQueryString());
                logger.error("==========HOST:[{}] uid:[{}] INVALID ACCOUNT ID[{}]", host, uid, token.getAccountId());
                return HttpCode2.ERR_AUTH_ACCOUNT_ID_CONFLICT;
            }

            //只有盒子的时候才校验DID
            if (token.getClientType() == TokenV2.TOKEN_FOR_DEVICE) {
                String did = HttpParameterTools.getParameter(httpRequest, HttpConstant.HTTP_ARG_DID);
                if (Strings.isNullOrEmpty(did) || !did.equalsIgnoreCase(token.getDid())) {
                    logger.error("==========HOST:[{}] MAC:[{}] Invalid Mac[{}]", host, did, token.getDid());
                    return HttpCode2.ERR_AUTH_INVALID_DID;
                }
            }

            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_ACCOUNT_ID, token.getAccountId());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_DID, token.getDid());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_UID, token.getDid2());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_TS, token.getGenerateTimestamp());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_UPROP, token.getUserProps());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_TOKEN, getToken(httpRequest));
            httpRequest.setAttribute(HttpConstant.HTTP_ARG_ACCOUNT_TYPE, token.get(AuthTokenFiled.ACCOUNT_TYPE.name()));
            if (token.getVersion() >= TokenV2.V20180808) {
                httpRequest.setAttribute(HttpConstant.HTTP_ARG_CLIENT_SESSION, token.get(AuthTokenFiled.CLIENT_SESSION.name()));
            }




            return HttpCode2.OK;
        } else {
            logger.error("==========HOST:[{}] Invalid token", host);
            return HttpCode2.ERR_AUTH_INVALID_TOKEN;
        }
    }

    @Override
    public Object getTokenObj(HttpServletRequest httpRequest, byte[] bytes) {
        return TokenV2.valueOf(bytes);
    }

    @Override
    public Set<String> getExcludedSet() {
        Set<String> excludedSet = super.getExcludedSet();

        excludedSet.add("/api/uac/login/v1");
        excludedSet.add("/api/uac/codeLogin/v1");

        excludedSet.add("/api/uac/activate/v1");
        excludedSet.add("/api/uac/activatedStatus/v1");


        excludedSet.add("/api/ns/*");
        excludedSet.add("/api/ping2");
        excludedSet.add("/api/uc/login4web/v1");
        excludedSet.add("/api/uc/login4app/v1");
        excludedSet.add("/api/uc/login4m/v1");


        excludedSet.add("/api/uc/register/v1");
        excludedSet.add("/api/uc/checkUser/v1");
        excludedSet.add("/api/uc/activateUser/v1");
        excludedSet.add("/api/uc/resetPassword/v1");
        excludedSet.add("/api/account/notice/setOnlineLimit/v1");
        return excludedSet;
    }
}
