package com.common.web.tools;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.valor.mfc.vms.api.model.common.response.ResponseStatus;
import com.valor.mfc.vms.common.tools.type.DateTimeTools;

import common.base.tools.encrypt.Base62Tools;
import common.config.tools.config.ConfigTools3;
import common.web.tools.http.model.response.WebApiBaseResponse;

import java.util.Date;
import java.util.List;
import java.util.Map;

public class ErrorMsgBuilder {

    private static final Long CONST_XOR = 10001311L;

    /**
     * encode device id
     *
     * @param : device id, general use device MAC address
     * @return
     */
    public static String encodeDeviceId(String deviceId) {
        try {
            if (Strings.isNullOrEmpty(deviceId)) {
                return "na";
            }

            long didValue = Long.valueOf(deviceId, 16);
            return Base62Tools.longToBase62(didValue ^ CONST_XOR);
        } catch (Exception e) {
            return "na";
        }
    }

    /**
     * decode device id
     *
     * @param encodeDeviceId
     * @return
     */
    public static String decodeDeviceId(String encodeDeviceId) {
//        long base62Str = Number62Tools._62to10(encodeDeviceId);
        long base62Str = Base62Tools.base62ToLong(encodeDeviceId);
        return Long.toHexString(base62Str ^ CONST_XOR);
    }


    /**
     * encode timestamp(unit:second)
     */
    public static String encodeTimestamp() {
        try {
//            return Number62Tools._10to62(System.currentTimeMillis() / 1000);
            return Base62Tools.longToBase62(System.currentTimeMillis() / 1000);
        } catch (Exception e) {
            return "na";
        }
    }

    /**
     * decode timestamp
     *
     * @param encodeTimestamp
     * @return
     */
    public static String decodeTimestamp(String encodeTimestamp) {
//        long ts = Number62Tools._62to10(encodeTimestamp);
        long ts = Base62Tools.base62ToLong(encodeTimestamp);
        return DateTimeTools.datetimeToString(new Date(ts * 1000));
    }


    /**
     * build error message
     *
     * @param language    : error description language,default is english
     * @param deviceId    : device id, general use device MAC address
     * @param retCode     : return code
     * @param errCode     : error code
     * @param defaultMsg: default error message,if msg not config
     * @return
     */
    public static String buildErrMsg(String language,
                                     String deviceId,
                                     int retCode,
                                     int errCode,
                                     String defaultMsg) {

        String langCode = Strings.nullToEmpty(language).toLowerCase();
        if (!langCode.equalsIgnoreCase("pt")
            && !langCode.equalsIgnoreCase("es")) {
            langCode = "en";
        }

        String configKey = String.format("%s.%s.%s", langCode, retCode, errCode);

        String configEnKey = String.format("%s.%s.%s", "en", retCode, errCode);

        String errDesc = ConfigTools3.getString(configKey);


        if (Strings.isNullOrEmpty(errDesc) && !"en".equalsIgnoreCase(langCode)) {

            errDesc = ConfigTools3.getString(configEnKey);


        }

        if (Strings.isNullOrEmpty(errDesc) && !Strings.isNullOrEmpty(defaultMsg)) {
            errDesc = defaultMsg;
        }

        if (Strings.isNullOrEmpty(errDesc)){
            errDesc="na";
        }

        String errMsg = String.format("%s.%s.%s.%s:%s",
            encodeDeviceId(deviceId), encodeTimestamp(), retCode, errCode, errDesc);
        return errMsg;
    }

    /**
     * Parse  error message
     * error message format : xxxx.xxxx.xxx.xxx
     *
     * @param errorMsg
     * @return
     */
    public static Map<String, String> parseErrMsg(String errorMsg) {
        Map<String, String> elementMap = Maps.newLinkedHashMap();
        if (Strings.isNullOrEmpty(errorMsg)) {
            return elementMap;
        }

        List<String> elementList = Splitter.on(".").splitToList(errorMsg);
        if (elementList.isEmpty()) {
            return elementMap;
        }

        String deviceId = "";
        String ts = "";
        int retCode = -1;
        int errCode = -1;

        for (int i = 0; i < elementList.size(); i++) {
            String value = elementList.get(i);
            if (i == 0) {
                deviceId = decodeDeviceId(value);
            }

            if (i == 1) {
                ts = decodeTimestamp(value);
            }

            if (i == 2) {
                retCode = Integer.parseInt(value);
            }

            if (i == 3) {
                errCode = Integer.parseInt(value);
            }
        }

        elementMap.put("Device Id", deviceId);
        elementMap.put("Timestamp", ts);
        elementMap.put("Module", retCode + "");
        elementMap.put("Error No", errCode + "");


        if (retCode > 0 && errCode > 0) {
            String enErrMsg = ConfigTools3.getString(String.format("en.%s.%s", retCode, errCode));
            if (!Strings.isNullOrEmpty(enErrMsg)) {
                elementMap.put("Error Message", enErrMsg);
            }
        }

        return elementMap;
    }

    /**
     * demp error msg
     *
     * @param errMsg
     * @return
     */
    public static String dump(String errMsg) {
        Map<String, String> errInfo = parseErrMsg(errMsg);
        StringBuilder sb = new StringBuilder();
        sb.append("\n*******************************\n");
        sb.append(Joiner.on("\n").withKeyValueSeparator(" : ").join(errInfo));
        sb.append("\n*******************************\n");
        return sb.toString();
    }

    /**
     * Set response status
     *
     * @param response
     * @param language
     * @param did
     * @param retCode
     * @param errCode
     * @param defaultMsg
     */
    public static void setResponse(ResponseStatus response,
                                   String language,
                                   String did,
                                   int retCode,
                                   int errCode,
                                   String defaultMsg) {
        response.setRetCode(retCode);
        response.setErrCode(errCode);
        response.setMessage(buildErrMsg(language, did, retCode, errCode, defaultMsg));
    }

    /**
     * Set response status
     *
     * @param response
     * @param language
     * @param did
     * @param retCode
     * @param errCode
     * @param defaultMsg
     */
    public static void setResponse(WebApiBaseResponse response,
                                   String language,
                                   String did,
                                   int retCode,
                                   int errCode,
                                   String defaultMsg) {
        response.setRetCode(retCode);
        response.setErrCode(errCode);
        response.setMessage(buildErrMsg(language, did, retCode, errCode, defaultMsg));
    }
}
