package com.mfc.es.api.client;

import com.google.common.base.Joiner;

import common.base.tools.stat.MetricTools;
import common.base.tools.stat.StatTools;

import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkProcessor;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by Frank.Huang on 2017/3/6.
 */
public class ESBulkProcessorListener implements BulkProcessor.Listener {
    private static final Logger logger = LoggerFactory.getLogger(ESIndexApi.class);
    @Override
    public void beforeBulk(long executionId, BulkRequest request) {
        logger.info("[ES-API][BULK-BEFORE]executionId[{}] request[number:({}) size:({})]",executionId,
                request.numberOfActions(),request.estimatedSizeInBytes());

        for (DocWriteRequest req : request.requests()) {
            if (req instanceof DocWriteRequest){
                DocWriteRequest dr = (DocWriteRequest) req;
                StatTools.addReq(Joiner.on("_").join("es",dr.index(),dr.type()));
            }
        }
    }

    @Override
    public void afterBulk(long executionId, BulkRequest request, BulkResponse response) {
        if (!response.hasFailures()){
            logger.info("[ES-API][BULK-AFTER]executionId[{}] request[number:({}) size:({})] response[Took:[{}] IngestTook:[{}]]",
                    executionId,
                    request.numberOfActions(),
                    request.estimatedSizeInBytes(),
                    response.getTook().getMillis(),
                    response.getIngestTookInMillis());

        }else{
            logger.error("[ES-API][BULK-AFTER]executionId[{}] request[number:({}) size:({})] response[Took:[{}] IngestTook:[{}]] Failure msg:{}",
                    executionId,request.numberOfActions(),
                    request.estimatedSizeInBytes(),
                    response.getTook().getMillis(),
                    response.getIngestTookInMillis()
                    ,response.buildFailureMessage());
        }

        MetricTools.logESTime("bulk", "bulk", request.numberOfActions(), response.getTook().getMillis());
        for (BulkItemResponse bis : response.getItems()){
            if (bis.isFailed()){
                StatTools.addErr(Joiner.on("_").join("es",bis.getIndex(),bis.getType()));
            }else {
                StatTools.addRsp(Joiner.on("_").join("es",bis.getIndex(),bis.getType()));
            }
        }
    }

    @Override
    public void afterBulk(long executionId, BulkRequest request, Throwable failure) {
        logger.error(String.format("[ES-API][BULK-AFTER][EXCEPTION]executionId[%s] request[number:(%s) size:(%s)]",executionId,request.numberOfActions(),request.estimatedSizeInBytes()), failure);

        for (DocWriteRequest req : request.requests()) {
            if (req instanceof DocWriteRequest){
                DocWriteRequest dr = (DocWriteRequest) req;
                StatTools.addErr(Joiner.on("_").join("es",dr.index(),dr.type()));
            }
        }
    }
}
