package com.mfc.es.api.client;

import com.mfc.es.api.model.client.ESProperties;
import com.mfc.es.api.model.service.base.IDocument;
import org.elasticsearch.action.bulk.BackoffPolicy;
import org.elasticsearch.action.bulk.BulkProcessor;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.List;

/**
 * Created by Frank.Huang on 2017/2/28.
 */
public class ESIndexApi extends ESApi {
    private static final Logger logger = LoggerFactory.getLogger(ESIndexApi.class);
    protected BulkProcessor bulkProcessor = null;

    public ESIndexApi(ESProperties esProperties) {
        super(esProperties);
        initBulkProcessor();
    }

    public IndexResponse updateIndex(String index, String type, IDocument doc) {
        if (!hasInit()) {
            return null;
        }

        String json = toJson(doc);
        if (!Strings.isNullOrEmpty(json)) {
            IndexRequest indexRequest = new IndexRequest(index, type, doc.getDocumentId());
            indexRequest.source(json, XContentType.JSON);
            try {
                return client.index(indexRequest);
            } catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }

        return null;
    }

    public BulkResponse bulkUpdateIndex(String index, String type, List<IDocument> docList) {
        BulkRequest bulkRequest = new BulkRequest();
        for (IDocument doc : docList) {
            String json = toJson(doc);
            if (!Strings.isNullOrEmpty(json)) {
                try {
                    bulkRequest.add(new IndexRequest(index, type, doc.getDocumentId()).source(json, XContentType.JSON));
                }catch (Exception e){
                    logger.error("bulkUpdateIndex bulkRequest error");
                    return null;
                }
            }
        }

        if (bulkRequest.numberOfActions() > 0) {
            try {
                return client.bulk(bulkRequest);
            } catch (IOException e) {
                logger.error("[bulkUpdateIndex][ERROR],ex:[{}]", e);
                return null;
            }
        } else {
            logger.error("[bulkUpdateIndex] numberOfActions<=0");
            return null;
        }
    }

    public void initBulkProcessor() {
        bulkProcessor = BulkProcessor.builder(client::bulkAsync, new ESBulkProcessorListener())
            .setBulkActions(properties.bulkSize())
            .setBulkSize(new ByteSizeValue(properties.bulkByteSize(), ByteSizeUnit.MB))
            .setFlushInterval(TimeValue.timeValueSeconds(properties.bulkFlushInterval()))
            .setConcurrentRequests(properties.bulkConcurrent())
            .setBackoffPolicy(BackoffPolicy.exponentialBackoff(TimeValue.timeValueMillis(properties.bulkBackoffInitialdelay()), properties.bulkBackoffRetry()))
            .build();

    }

    public void closeBulkProcessor() {
        bulkProcessor.close();
    }

    public void bulkProcessorUpsert(String index, String type, IDocument document) {
        if (document == null) {
            return;
        }

        String json = toJson(document);
        if (!Strings.isNullOrEmpty(json)) {
            bulkProcessor.add(new IndexRequest(index, type, document.getDocumentId()).source(json, XContentType.JSON));
        }
    }

    public void bulkProcessorRemove(String index, String type, String id) {
        bulkProcessor.add(new DeleteRequest(index, type, id));
    }

}

