package com.mfc.es.api.client;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mfc.es.api.model.client.ESProperties;
import com.mfc.es.api.model.service.cloud.MediaCloud;
import com.mfc.es.api.model.service.cloud.MediaCloudSearchIndex;
import com.mfc.es.api.model.service.media.*;
import com.mfc.es.api.model.service.person.Person;
import com.mfc.es.api.model.service.person.PersonSearchIndex;
import com.mfc.es.api.model.service.search.*;
import common.base.tools.stat.TimeStatisticsTools;
import common.config.tools.config.ConfigTools3;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.get.MultiGetItemResponse;
import org.elasticsearch.action.get.MultiGetRequest;
import org.elasticsearch.action.get.MultiGetResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.Response;
import org.elasticsearch.common.ParseField;
import org.elasticsearch.common.xcontent.ContextParser;
import org.elasticsearch.common.xcontent.DeprecationHandler;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.NestedQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.bucket.adjacency.AdjacencyMatrixAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.adjacency.ParsedAdjacencyMatrix;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.composite.ParsedComposite;
import org.elasticsearch.search.aggregations.bucket.filter.FilterAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.filter.FiltersAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.filter.ParsedFilter;
import org.elasticsearch.search.aggregations.bucket.filter.ParsedFilters;
import org.elasticsearch.search.aggregations.bucket.geogrid.GeoGridAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.geogrid.ParsedGeoHashGrid;
import org.elasticsearch.search.aggregations.bucket.global.GlobalAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.global.ParsedGlobal;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.HistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedHistogram;
import org.elasticsearch.search.aggregations.bucket.missing.MissingAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.missing.ParsedMissing;
import org.elasticsearch.search.aggregations.bucket.nested.NestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedNested;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedReverseNested;
import org.elasticsearch.search.aggregations.bucket.nested.ReverseNestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.range.*;
import org.elasticsearch.search.aggregations.bucket.sampler.InternalSampler;
import org.elasticsearch.search.aggregations.bucket.sampler.ParsedSampler;
import org.elasticsearch.search.aggregations.bucket.significant.ParsedSignificantLongTerms;
import org.elasticsearch.search.aggregations.bucket.significant.ParsedSignificantStringTerms;
import org.elasticsearch.search.aggregations.bucket.significant.SignificantLongTerms;
import org.elasticsearch.search.aggregations.bucket.significant.SignificantStringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.*;
import org.elasticsearch.search.aggregations.metrics.avg.AvgAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.avg.ParsedAvg;
import org.elasticsearch.search.aggregations.metrics.cardinality.CardinalityAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.cardinality.ParsedCardinality;
import org.elasticsearch.search.aggregations.metrics.geobounds.GeoBoundsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.geobounds.ParsedGeoBounds;
import org.elasticsearch.search.aggregations.metrics.geocentroid.GeoCentroidAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.geocentroid.ParsedGeoCentroid;
import org.elasticsearch.search.aggregations.metrics.max.MaxAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.max.ParsedMax;
import org.elasticsearch.search.aggregations.metrics.min.MinAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.min.ParsedMin;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.InternalHDRPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.InternalHDRPercentiles;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.ParsedHDRPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.ParsedHDRPercentiles;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.InternalTDigestPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.InternalTDigestPercentiles;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.ParsedTDigestPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.ParsedTDigestPercentiles;
import org.elasticsearch.search.aggregations.metrics.scripted.ParsedScriptedMetric;
import org.elasticsearch.search.aggregations.metrics.scripted.ScriptedMetricAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.stats.ParsedStats;
import org.elasticsearch.search.aggregations.metrics.stats.StatsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.stats.extended.ExtendedStatsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.stats.extended.ParsedExtendedStats;
import org.elasticsearch.search.aggregations.metrics.sum.ParsedSum;
import org.elasticsearch.search.aggregations.metrics.sum.SumAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.tophits.ParsedTopHits;
import org.elasticsearch.search.aggregations.metrics.tophits.TopHitsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.valuecount.ParsedValueCount;
import org.elasticsearch.search.aggregations.metrics.valuecount.ValueCountAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.InternalSimpleValue;
import org.elasticsearch.search.aggregations.pipeline.ParsedSimpleValue;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.InternalBucketMetricValue;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.ParsedBucketMetricValue;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.percentile.ParsedPercentilesBucket;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.percentile.PercentilesBucketPipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.ParsedStatsBucket;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.StatsBucketPipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.extended.ExtendedStatsBucketPipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.extended.ParsedExtendedStatsBucket;
import org.elasticsearch.search.aggregations.pipeline.derivative.DerivativePipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.derivative.ParsedDerivative;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.*;
import org.elasticsearch.search.suggest.Suggest;
import org.elasticsearch.search.suggest.completion.CompletionSuggestion;
import org.elasticsearch.search.suggest.phrase.PhraseSuggestion;
import org.elasticsearch.search.suggest.term.TermSuggestion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

/**
 * Created by Frank.Huang on 2017/3/16.
 */
public class ESSearchApi extends ESApi {
    private static final Logger logger = LoggerFactory.getLogger(ESSearchApi.class);
    private static final DeprecationHandler DEPRECATION_HANDLER = new DeprecationHandler() {
        @Override
        public void usedDeprecatedName(String usedName, String modernName) {}
        @Override
        public void usedDeprecatedField(String usedName, String replacedWith) {}
    };

    public ESSearchApi(ESProperties esProperties) {
        super(esProperties);
    }

    public <T> ESApiResponse<T> search(ESSearchParameter parameter) {

        long beginTime=System.currentTimeMillis();
        String queryJson=parseParameter(parameter);
        boolean printJson = ConfigTools3.getBoolean("mfc.es.search.req.log.enabled", false);
        if (printJson){
            logger.info("[ESSearchApi][search] queryJson[{}]",queryJson);
        }
        logger.debug("[ESSearchApi][search] queryJson[{}]",queryJson);
        logger.debug("[ESSearchApi][search] parseParameter[{}]",System.currentTimeMillis()-beginTime);
        try {

            HttpEntity entity = new NStringEntity(queryJson, ContentType.APPLICATION_JSON);
            Request request = new Request("POST","/"+parameter.getIndex()+"/"+parameter.getType()+"/_search");
            request.setEntity(entity);
            Response response = lowClient.performRequest(request);
            logger.debug("[ESSearchApi][search] performRequest[{}]",System.currentTimeMillis()-beginTime);

            beginTime=System.currentTimeMillis();
            NamedXContentRegistry registry = new NamedXContentRegistry(Stream.of(getDefaultNamedXContents().stream()).flatMap(Function.identity()).collect(toList()));
            SearchResponse searchResponse=SearchResponse.fromXContent(JsonXContent.jsonXContent.createParser(registry, DEPRECATION_HANDLER, response.getEntity().getContent()));
            logger.debug("[ESSearchApi][search] parseContent[{}]",System.currentTimeMillis()-beginTime);

            beginTime=System.currentTimeMillis();
            ESApiResponse<T> esApiResponse = toResponse(searchResponse);
            logger.debug("[ESSearchApi][search] parseObject[{}]",System.currentTimeMillis()-beginTime);

            return esApiResponse;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    private String parseParameter(ESSearchParameter parameter)
    {
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        List<ESNestedFilterBuilder> esNestedFilterBuilders = Lists.newArrayList();
        for (IESFilterBuilder e : parameter.getFilters()) {
            QueryBuilder queryBuilder = null;
            EESFilterType esFilterType = EESFilterType.MUST;
            if (e instanceof ESTermFilterBuilder) {
                ESTermFilterBuilder termFilter = (ESTermFilterBuilder) e;
                if (termFilter.getValue() == null && termFilter.getValues() == null) {
                    continue;
                }

                if (termFilter.isMultiValue()) {
                    queryBuilder = QueryBuilders.termsQuery(termFilter.getFieldName(), termFilter.getValues());
                } else {
                    queryBuilder = QueryBuilders.termQuery(termFilter.getFieldName(), termFilter.getValue());
                }
                esFilterType = termFilter.getFilterType();
            } else if (e instanceof ESRangeFilterBuilder) {
                ESRangeFilterBuilder rangeFilter = (ESRangeFilterBuilder) e;

                queryBuilder = QueryBuilders.rangeQuery(rangeFilter.getFieldName())
                        .from(rangeFilter.getFrom())
                        .to(rangeFilter.getTo())
                        .includeLower(rangeFilter.isIncludeLower())
                        .includeUpper(rangeFilter.isIncludeUpper());

                esFilterType = rangeFilter.getFilterType();
            }else if(e instanceof ESNestedFilterBuilder){
                ESNestedFilterBuilder nestedFilterBuilder = (ESNestedFilterBuilder)e;
                queryBuilder = QueryBuilders.nestedQuery(nestedFilterBuilder.getPath(),nestedFilterBuilder.getBoolQueryBuilder(),nestedFilterBuilder.getScoreMode());
                esFilterType = nestedFilterBuilder.getFilterType();
                esNestedFilterBuilders.add(nestedFilterBuilder);
            }
            if (queryBuilder != null) {
                switch (esFilterType) {
                    case MUST:
                        boolQueryBuilder.must(queryBuilder);
                        break;
                    case MUST_NOT:
                        boolQueryBuilder.mustNot(queryBuilder);
                        break;
                    case SHOULD:
                        boolQueryBuilder.should(queryBuilder);
                        break;
                    default:
                        boolQueryBuilder.must(queryBuilder);
                }
            }
        }

        if (!parameter.isShowAll() && parameter.getShowUntilDate() > 0) {
            BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery();
            queryBuilder.should(QueryBuilders.rangeQuery("releaseDateInt").lt(parameter.getShowUntilDate()));
            queryBuilder.should(QueryBuilders.termQuery("connected", 1));
            boolQueryBuilder.must(queryBuilder);
        }

        BoolQueryBuilder filterBuilder = QueryBuilders.boolQuery();
        filterBuilder.filter(boolQueryBuilder);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.query(filterBuilder);
        searchSourceBuilder.explain(false);
        searchSourceBuilder.from(parameter.getFrom());
        searchSourceBuilder.size(parameter.getSize());

        parameter.getOrders().forEach(e -> {
            SortOrder order =  e.isAscending() ? SortOrder.ASC : SortOrder.DESC;
            if (e.getSource().equals(ScoreSortBuilder.NAME)) {
                ScoreSortBuilder scoreSortBuilder = SortBuilders.scoreSort().order(order);
                searchSourceBuilder.sort(scoreSortBuilder);
            }else{
                FieldSortBuilder fieldSortBuilder =SortBuilders.fieldSort(e.getSource()).order(order);
                if(StringUtils.isNotEmpty(e.getNestedPath())){
                    fieldSortBuilder.setNestedPath(e.getNestedPath());
                    List<ESNestedFilterBuilder> sortESNestedFilterBuilders = esNestedFilterBuilders.stream().filter(nf->nf.getPath().equals(e.getNestedPath())).collect(toList());
                    BoolQueryBuilder sortBoolQueryBuilder = QueryBuilders.boolQuery();
                    sortESNestedFilterBuilders.forEach(snf->{
                        if (snf.getFilterType() != null) {
                            switch (snf.getFilterType()) {
                                case MUST:
                                    sortBoolQueryBuilder.must(snf.getBoolQueryBuilder());
                                    break;
                                case MUST_NOT:
                                    sortBoolQueryBuilder.mustNot(snf.getBoolQueryBuilder());
                                    break;
                                case SHOULD:
                                    sortBoolQueryBuilder.should(snf.getBoolQueryBuilder());
                                    break;
                                default:
                                    sortBoolQueryBuilder.must(snf.getBoolQueryBuilder());
                            }
                        }
                    });
                    fieldSortBuilder.setNestedFilter(sortBoolQueryBuilder);
                }
                searchSourceBuilder.sort(fieldSortBuilder);
            }
        });

        return searchSourceBuilder.toString();
    }

    private <T> ESApiResponse<T> toResponse(SearchResponse searchResponse) {
        ESApiResponse<T> response = new ESApiResponse<T>();
        if (searchResponse != null) {
            response.setTookTime(searchResponse.getTook().getMillis());
            response.setHitTotal(searchResponse.getHits().getTotalHits());
            response.setHitCount(searchResponse.getHits().getHits().length);

            long curTS=System.currentTimeMillis();
            for (SearchHit searchHit : searchResponse.getHits().getHits()) {
                Class<T> valueType = getValueType(searchHit.getType());
                T v = fromJsonStr(searchHit.getSourceAsString(), valueType);
                response.getSource().add(v);
            }
            TimeStatisticsTools.addTime(String.format("ES-SEARCH-API:SEARCH:JSON"), searchResponse.getHits().getHits().length, System.currentTimeMillis() - curTS);
        }

        return response;
    }

    private <T> ESApiResponse<T> toResponse(MultiGetResponse multiGetResponse) {
        ESApiResponse<T> response = new ESApiResponse<T>();

        if (multiGetResponse != null) {
            for (MultiGetItemResponse itemResponse : multiGetResponse.getResponses()) {
                GetResponse getResponse = itemResponse.getResponse();
                if (getResponse.isExists()) {
                    Class<T> valueType = getValueType(getResponse.getType());
                    T v = fromJsonStr(getResponse.getSourceAsString(), valueType);
                    response.getSource().add(v);
                }
            }
        }

        return response;
    }

    public <T> Class<T> getValueType(String type) {
        switch (type) {
            case ESConstants.TYPE_MEDIA_BASE:
                return (Class<T>) MediaBase.class;
            case ESConstants.TYPE_MEDIA_SUMMARY:
                return (Class<T>) MediaSummary.class;
            case ESConstants.TYPE_MEDIA_DETAIL:
                return (Class<T>) MediaDetail.class;
            case ESConstants.TYPE_MEDIA_USER_REVIEW:
                return (Class<T>) MediaUserReview.class;
            case ESConstants.TYPE_MEDIA_SYNOPSIS:
                return (Class<T>) MediaSynopsis.class;
            case ESConstants.TYPE_MEDIA_PARENTS_GUIDE:
                return (Class<T>) MediaParentsGuide.class;
            case ESConstants.TYPE_MEDIA_IMAGE:
                return (Class<T>) MediaImage.class;
            case ESConstants.TYPE_PERSON:
                return (Class<T>) Person.class;
            case ESConstants.TYPE_MEDIA_CLOUD:
                return (Class<T>) MediaCloud.class;
            case ESConstants.TYPE_MEDIA_AWARD:
                return (Class<T>) MediaAward.class;
            case ESConstants.TYPE_MEDIA_SEARCHINDEX:
                return (Class<T>) MediaSearchIndex.class;
            case ESConstants.TYPE_MEDIA_CLOUD_SEARCHINDEX:
                return (Class<T>) MediaCloudSearchIndex.class;
            case ESConstants.TYPE_PERSON_SEARCHINDEX:
                return (Class<T>) PersonSearchIndex.class;
            default:
                return null;
        }
    }

    public <T> ESApiResponse<T> getDocumentByIds(String index, String type, Set<String> docIds) {
        MultiGetRequest mgetRequest = new MultiGetRequest();
        docIds.forEach(e -> {
            mgetRequest.add(index, type, e);
        });

        try {
            MultiGetResponse multiGetResponse = client.multiGet(mgetRequest);
            return toResponse(multiGetResponse);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    private List<NamedXContentRegistry.Entry> getDefaultNamedXContents() {
        Map<String, ContextParser<Object, ? extends Aggregation>> map = new HashMap<>();
        map.put(CardinalityAggregationBuilder.NAME, (p, c) -> ParsedCardinality.fromXContent(p, (String) c));
        map.put(InternalHDRPercentiles.NAME, (p, c) -> ParsedHDRPercentiles.fromXContent(p, (String) c));
        map.put(InternalHDRPercentileRanks.NAME, (p, c) -> ParsedHDRPercentileRanks.fromXContent(p, (String) c));
        map.put(InternalTDigestPercentiles.NAME, (p, c) -> ParsedTDigestPercentiles.fromXContent(p, (String) c));
        map.put(InternalTDigestPercentileRanks.NAME, (p, c) -> ParsedTDigestPercentileRanks.fromXContent(p, (String) c));
        map.put(PercentilesBucketPipelineAggregationBuilder.NAME, (p, c) -> ParsedPercentilesBucket.fromXContent(p, (String) c));
        map.put(MinAggregationBuilder.NAME, (p, c) -> ParsedMin.fromXContent(p, (String) c));
        map.put(MaxAggregationBuilder.NAME, (p, c) -> ParsedMax.fromXContent(p, (String) c));
        map.put(SumAggregationBuilder.NAME, (p, c) -> ParsedSum.fromXContent(p, (String) c));
        map.put(AvgAggregationBuilder.NAME, (p, c) -> ParsedAvg.fromXContent(p, (String) c));
        map.put(ValueCountAggregationBuilder.NAME, (p, c) -> ParsedValueCount.fromXContent(p, (String) c));
        map.put(InternalSimpleValue.NAME, (p, c) -> ParsedSimpleValue.fromXContent(p, (String) c));
        map.put(DerivativePipelineAggregationBuilder.NAME, (p, c) -> ParsedDerivative.fromXContent(p, (String) c));
        map.put(InternalBucketMetricValue.NAME, (p, c) -> ParsedBucketMetricValue.fromXContent(p, (String) c));
        map.put(StatsAggregationBuilder.NAME, (p, c) -> ParsedStats.fromXContent(p, (String) c));
        map.put(StatsBucketPipelineAggregationBuilder.NAME, (p, c) -> ParsedStatsBucket.fromXContent(p, (String) c));
        map.put(ExtendedStatsAggregationBuilder.NAME, (p, c) -> ParsedExtendedStats.fromXContent(p, (String) c));
        map.put(ExtendedStatsBucketPipelineAggregationBuilder.NAME,
                (p, c) -> ParsedExtendedStatsBucket.fromXContent(p, (String) c));
        map.put(GeoBoundsAggregationBuilder.NAME, (p, c) -> ParsedGeoBounds.fromXContent(p, (String) c));
        map.put(GeoCentroidAggregationBuilder.NAME, (p, c) -> ParsedGeoCentroid.fromXContent(p, (String) c));
        map.put(HistogramAggregationBuilder.NAME, (p, c) -> ParsedHistogram.fromXContent(p, (String) c));
        map.put(DateHistogramAggregationBuilder.NAME, (p, c) -> ParsedDateHistogram.fromXContent(p, (String) c));
        map.put(StringTerms.NAME, (p, c) -> ParsedStringTerms.fromXContent(p, (String) c));
        map.put(LongTerms.NAME, (p, c) -> ParsedLongTerms.fromXContent(p, (String) c));
        map.put(DoubleTerms.NAME, (p, c) -> ParsedDoubleTerms.fromXContent(p, (String) c));
        map.put(MissingAggregationBuilder.NAME, (p, c) -> ParsedMissing.fromXContent(p, (String) c));
        map.put(NestedAggregationBuilder.NAME, (p, c) -> ParsedNested.fromXContent(p, (String) c));
        map.put(ReverseNestedAggregationBuilder.NAME, (p, c) -> ParsedReverseNested.fromXContent(p, (String) c));
        map.put(GlobalAggregationBuilder.NAME, (p, c) -> ParsedGlobal.fromXContent(p, (String) c));
        map.put(FilterAggregationBuilder.NAME, (p, c) -> ParsedFilter.fromXContent(p, (String) c));
        map.put(InternalSampler.PARSER_NAME, (p, c) -> ParsedSampler.fromXContent(p, (String) c));
        map.put(GeoGridAggregationBuilder.NAME, (p, c) -> ParsedGeoHashGrid.fromXContent(p, (String) c));
        map.put(RangeAggregationBuilder.NAME, (p, c) -> ParsedRange.fromXContent(p, (String) c));
        map.put(DateRangeAggregationBuilder.NAME, (p, c) -> ParsedDateRange.fromXContent(p, (String) c));
        map.put(GeoDistanceAggregationBuilder.NAME, (p, c) -> ParsedGeoDistance.fromXContent(p, (String) c));
        map.put(FiltersAggregationBuilder.NAME, (p, c) -> ParsedFilters.fromXContent(p, (String) c));
        map.put(AdjacencyMatrixAggregationBuilder.NAME, (p, c) -> ParsedAdjacencyMatrix.fromXContent(p, (String) c));
        map.put(SignificantLongTerms.NAME, (p, c) -> ParsedSignificantLongTerms.fromXContent(p, (String) c));
        map.put(SignificantStringTerms.NAME, (p, c) -> ParsedSignificantStringTerms.fromXContent(p, (String) c));
        map.put(ScriptedMetricAggregationBuilder.NAME, (p, c) -> ParsedScriptedMetric.fromXContent(p, (String) c));
        map.put(IpRangeAggregationBuilder.NAME, (p, c) -> ParsedBinaryRange.fromXContent(p, (String) c));
        map.put(TopHitsAggregationBuilder.NAME, (p, c) -> ParsedTopHits.fromXContent(p, (String) c));
        map.put(CompositeAggregationBuilder.NAME, (p, c) -> ParsedComposite.fromXContent(p, (String) c));
        List<NamedXContentRegistry.Entry> entries = map.entrySet().stream()
                .map(entry -> new NamedXContentRegistry.Entry(Aggregation.class, new ParseField(entry.getKey()), entry.getValue()))
                .collect(Collectors.toList());
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(TermSuggestion.NAME),
                (parser, context) -> TermSuggestion.fromXContent(parser, (String)context)));
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(PhraseSuggestion.NAME),
                (parser, context) -> PhraseSuggestion.fromXContent(parser, (String)context)));
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(CompletionSuggestion.NAME),
                (parser, context) -> CompletionSuggestion.fromXContent(parser, (String)context)));
        return entries;
    }

    public static void main(String[] args) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_DETAIL).type(ESConstants.TYPE_MEDIA_DETAIL).size(10000).add(ESTermFilterBuilder.builder("seriesId").setValue(19870832));
//        String json=parseParameter(parameter);
//        System.out.println(json);
    }
}
