package com.valor.mfc.vms.common.database.tool.configruation;

import com.google.common.base.Strings;
import com.valor.mfc.vms.common.database.model.TableShardingParams;
import common.config.tools.config.ConfigAESTools;
import common.config.tools.config.ConfigTools3;
import io.shardingsphere.core.api.ShardingDataSourceFactory;
import io.shardingsphere.core.api.config.ShardingRuleConfiguration;
import io.shardingsphere.core.api.config.TableRuleConfiguration;
import io.shardingsphere.core.api.config.strategy.InlineShardingStrategyConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;

import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.*;

/**
 * @author Hunter Chen
 * created on: 2018-08-28
 * <p>
 * Abstract database config, with table sharding rules
 * Extend this class if table sharding is needed
 */
public abstract class AbstractDBCfgWithSharding {

    /**
     * Initiate a desired datasource pool, eg, Hiraki, Druid, and configure it properly
     *
     * @return
     */
    protected abstract DataSource createDataSouce();

    /**
     * Create a list of TableShardingParams to designate the ways tables will be sharded
     *
     * @return
     */
    protected abstract List<TableShardingParams> configureTableShardingParams();

    /**
     * return the database name of the tables to be sharded
     *
     * @return
     */
    protected abstract String configureDatabaseName();

    // whether to print real sql sent to dataBase after table sharding
    protected Boolean showSql=false;

    private List<TableRuleConfiguration> getTableRules() {
        List<TableRuleConfiguration> rules = new ArrayList<>();
        List<TableShardingParams> tableShardingParams = configureTableShardingParams();
        if (tableShardingParams != null && tableShardingParams.size() > 0) {
            for (TableShardingParams params : tableShardingParams) {
                TableRuleConfiguration rule = new TableRuleConfiguration();
                rule.setLogicTable(params.getTableName());
                rule.setActualDataNodes(params.getActualDataNodesExpression());
                rule.setTableShardingStrategyConfig(new InlineShardingStrategyConfiguration(params.getShardingColumn(), params.getStrategyExpression()));
                rules.add(rule);
            }
        }
        return rules;
    }

    @Primary
    @Bean(name = "shardingDataSource")
    public DataSource getShardingDataSource() throws SQLException {
        ShardingRuleConfiguration shardingRuleConfig = new ShardingRuleConfiguration();
        List<TableRuleConfiguration> tableRules = getTableRules();
        if (tableRules != null && tableRules.size() > 0) {
            shardingRuleConfig.getTableRuleConfigs().addAll(tableRules);
        }
        Properties p = new Properties();
        if (showSql) {
            p.setProperty("sql.show", "true");
        }
        return ShardingDataSourceFactory.createDataSource(createDataSourceMap(), shardingRuleConfig, new HashMap<String, Object>(), p);
    }

    private Map<String, DataSource> createDataSourceMap() {
        Map<String, DataSource> result = new HashMap<>(1, 1);
        DataSource dataSouce = createDataSouce();
        if (dataSouce == null) {
            throw new RuntimeException("Datasource is not properly configured!");
        }
        result.put(configureDatabaseName(), dataSouce);
        return result;
    }

    /**
     * To decrypt messages, like username and password
     *
     * @param configKey
     * @param cryptKey
     * @return
     */
    protected String getDBConfig(String configKey, String cryptKey) {
        if (Strings.isNullOrEmpty(cryptKey)) {
            return ConfigTools3.getString(configKey);
        } else {
            return ConfigAESTools.getAESString(configKey, cryptKey);
        }
    }
}
