package com.valor.mfc.vms.common.cache.redis;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.*;
import org.springframework.data.redis.serializer.RedisSerializer;
import redis.clients.jedis.JedisCommands;

import java.util.*;
import java.util.concurrent.TimeUnit;

public class RedisCache implements RedisCacheable {
    private static final Logger logger = LoggerFactory.getLogger(RedisCache.class);

    public RedisCache(RedisTemplate redisTemplate)
    {
        this.redisTemplate = redisTemplate;
    }

    private RedisTemplate redisTemplate;


    private void expLog(String fun, Exception ex){
    }


    @Override
    public void del(String... keys) {
        try {
            if (keys != null && keys.length > 0) {
                if (keys.length == 1) {
                    redisTemplate.delete(keys[0]);
                } else {
                    redisTemplate.delete(Arrays.asList(keys));
                }
            }
        } catch (Exception ex) {
            logger.error("del(),exception:{}", ex);
            expLog("del()",ex);
        }
    }

    public void delete(Collection<String> keys){
        try {
            redisTemplate.delete(keys);
        } catch (Exception e) {
            logger.error("delete keys["+keys+"] exception",e);

        }
    }

    @Override
    public Boolean exists(String key) {
        try {
            return redisTemplate.hasKey(key);
        } catch (Exception ex) {
            logger.error("exists(),exception:{}", ex);
            expLog("exists()",ex);
            return null;
        }

    }

    @Override
    public void expire(String key, int seconds) {
        try {
            if (seconds > 0) {
                redisTemplate.expire(key, seconds, TimeUnit.SECONDS);
            }
        } catch (Exception ex) {
            logger.error("expire(),exception:{}", ex);
            expLog("expire()",ex);
        }
    }

    @Override
    public void pExpire(String key, int milliseconds) {
        try {
            if (milliseconds > 0) {
                redisTemplate.expire(key, milliseconds, TimeUnit.MILLISECONDS);
            }
        } catch (Exception ex) {
            logger.error("pExpire(),exception:{}", ex);
            expLog("pExpire()",ex);
        }
    }

    @Override
    public Set<String> keys(String pattern) {
        try {
            Set<String> keysSet = redisTemplate.keys(pattern);
            return keysSet;
        } catch (Exception ex) {
            logger.error("keys(),exception:{}", ex);
            expLog("keys()",ex);
            return null;
        }
    }

    @Override
    public void rename(String key, String newKey) {
        try {
            redisTemplate.rename(key, newKey);
        } catch (Exception ex) {
            logger.error("rename(),exception:{}", ex);
            expLog("rename()",ex);
        }
    }

    @Override
    public Long decr(String key) {
        try {
            return redisTemplate.opsForValue().increment(key, -1);
        } catch (Exception ex) {
            logger.error("decr(),exception:{}", ex);
            expLog("decr()",ex);
            return null;
        }
    }

    @Override
    public Long decrBy(String key, long decrement) {
        try {
            return redisTemplate.opsForValue().increment(key, -decrement);
        } catch (Exception ex) {
            logger.error("decrBy(),exception:{}", ex);
            expLog("decrBy()",ex);
            return null;
        }
    }

    @Override
    public Long incr(String key) {
        try {
            return redisTemplate.opsForValue().increment(key, 1);
        } catch (Exception ex) {
            logger.error("incr(),exception:{}", ex);
            expLog("incr()",ex);
            return null;
        }
    }

    @Override
    public Long incrBy(String key, long decrement) {
        try {
            return redisTemplate.opsForValue().increment(key, decrement);
        } catch (Exception ex) {
            logger.error("incrBy(),exception:{}", ex);
            expLog("incrBy()",ex);
            return null;
        }
    }

    @Override
    public  <V> V get(String key) {
        try {
            Object obj = redisTemplate.opsForValue().get(key);
            return (V)obj;
        } catch (Exception ex) {
            logger.error("get(),exception:{}", ex);
            expLog("get()",ex);
            return null;
        }
    }

    @Override
    public  <V> V getWithException(String key) {
        try {
            Object obj = redisTemplate.opsForValue().get(key);
            return (V)obj;
        } catch (Exception ex) {
            logger.error("get(),exception:{}", ex);
            expLog("get()",ex);
            throw ex;
        }
    }

    @Override
    public <V> V getSet(String key, V value) {
        try {
            return (V) redisTemplate.opsForValue().getAndSet(key, value);
        } catch (Exception ex) {
            logger.error("getSet(),exception:{}", ex);
            expLog("getSet()",ex);
            return null;
        }
    }

    @Override
    public <V> List<V> mGet(String... keys) {
        try {
            List<V> objs = redisTemplate.opsForValue().multiGet(Arrays.asList(keys));
//            int fireCount = 0;
//            if (objs != null){
//                for(int i=0;i<objs.size();i++){
//                    if (objs.get(i) != null){
//                        fireCount++;
//                    }
//                }
//            }
            return objs;
        } catch (Exception ex) {
            logger.error("mGet(),exception:{}", ex);
            expLog("mGet()",ex);
            return null;
        }
    }

    public <V> Map<String,V> mGet(Set<String> keySet) {
        try {
            List<String> keys = new ArrayList<>(keySet);
            List<V> objs = redisTemplate.opsForValue().multiGet(keys);
//            int fireCount = 0;
//            if (objs != null){
//                for(int i=0;i<objs.size();i++){
//                    if (objs.get(i) != null){
//                        fireCount++;
//                    }
//                }
//            }
            Map<String,V> resultMap = new HashMap<>();
            for(int i=0;i<keys.size();i++){
                V v = objs.get(i);

//                if (v == null){
//                    continue;
//                }
                resultMap.put(keys.get(i), v);
            }
            return resultMap;
        } catch (Exception ex) {
            logger.error("mGet(),exception:{}", ex);
            expLog("mGet()",ex);
            return null;
        }
    }

    public <V> Map<String,V> mGetWithoutNull(Set<String> keySet) {
        try {
            List<String> keys = new ArrayList<>(keySet);
            List<V> objs = (List<V>)redisTemplate.opsForValue().multiGet(keys);
            Map<String,V> resultMap = new HashMap<>();
            for(int i=0;i<keys.size();i++){
                V v = objs.get(i);
                if (v == null){
                    continue;
                }
                resultMap.put(keys.get(i), v);
            }
            return resultMap;
        } catch (Exception ex) {
            logger.error("mGet(),exception ", ex);
            return null;
        }
    }

    @Override
    public <V> void mSet(Map<String, V> keyValueMap) {
        try {
            redisTemplate.opsForValue().multiSet(keyValueMap);
        } catch (Exception ex) {
            logger.error("mSet(),exception:{}", ex);
            expLog("mSet()",ex);
        }
    }

    @Override
    public <V> void mSetNx(Map<String, V> keyValueMap) {
        try {
            redisTemplate.opsForValue().multiSetIfAbsent(keyValueMap);
        } catch (Exception ex) {
            logger.error("mSetNx(),exception:{}", ex);
            expLog("mSetNx()",ex);
        }
    }

    @Override
    public <V> void set(String key, V value) {
        try {
            redisTemplate.opsForValue().set(key, value);
        } catch (Exception ex) {
            logger.error("set(),exception:{}", ex);
            expLog("set()",ex);
        }
    }

    @Override
    public <V> void setEx(String key, V value, long seconds) {
        try {
            //改变了命令的默认行为，以便与之前的代码兼容
            if (seconds > 0) {
                redisTemplate.opsForValue().set(key, value, seconds, TimeUnit.SECONDS);
            } else {
                redisTemplate.opsForValue().set(key, value);
            }
        } catch (Exception ex) {
            logger.error("setEx(),exception:{}", ex);
            expLog("setEx()",ex);
        }
    }

    @Override
    public <V> void pipSetEx(Map<String, V> map, long seconds) {
        try {
            Object object = redisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    for (Map.Entry<String,V> e : map.entrySet()) {
                        String key = e.getKey();
                        V value = e.getValue();
                        RedisSerializer redisSerializer = redisTemplate.getValueSerializer();
                        if (seconds > 0){
                            redisConnection.setEx(key.getBytes(), seconds,redisSerializer.serialize(value));
                        }
                        else{
                            redisConnection.set(key.getBytes(),redisSerializer.serialize(value));
                        }
                    }
                    return null;
                }
            }, redisTemplate.getValueSerializer());
        } catch (Exception ex) {
            logger.error("pipHashSetEx(),exception:{}", ex);
            expLog("pipHashSetEx()",ex);
        }

    }


    @Override
    public <V> void pipSetExs(List<Map<String, V>> maps, long seconds) {
        try {
            Object object = redisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
                    for (int i=0;i<maps.size();i++){
                        Map<String, V> map = maps.get(i);
                        for (Map.Entry<String,V> e : map.entrySet()) {
                            String key = e.getKey();
                            V value = e.getValue();
                            RedisSerializer redisSerializer = redisTemplate.getValueSerializer();
                            if (seconds > 0){
                                redisConnection.setEx(key.getBytes(), seconds,redisSerializer.serialize(value));
                            }
                            else{
                                redisConnection.set(key.getBytes(),redisSerializer.serialize(value));
                            }
                        }
                    }
                    return null;
                }
            }, redisTemplate.getValueSerializer());
        } catch (Exception ex) {
            logger.error("pipSetExs(),exception:{}", ex);
            expLog("pipSetExs()",ex);
        }

    }


    @Override
    public <V> void pSetEx(String key, V value, long milliseconds) {
        try {
            if (milliseconds > 0) {
                redisTemplate.opsForValue().set(key, value, milliseconds, TimeUnit.MILLISECONDS);
            } else {
                redisTemplate.opsForValue().set(key, value);
            }
        } catch (Exception ex) {
            logger.error("pSetEx(),exception:{}", ex);
            expLog("pSetEx()",ex);
        }
    }

    @Override
    public <V> void setNx(String key, V value) {
        try {
            redisTemplate.opsForValue().setIfAbsent(key, value);
        } catch (Exception ex) {
            logger.error("setNx(),exception:{}", ex);
            expLog("setNx()",ex);
        }
    }

    public Boolean setNxEx(String key, String value, long seconds) {
        try {
            RedisCallback<String> callback = (connection) -> {
                JedisCommands commands = (JedisCommands) connection.getNativeConnection();
                return commands.set(key, value, "NX", "EX", seconds);
            };
            Object result = redisTemplate.execute(callback);
            if (result == null){
                return false;
            }
            else{
                return true;
            }
        } catch (Exception ex) {
            try {
                redisTemplate.delete(key);
            }
            catch (Exception e){
            }
            logger.error("setNxEx(),exception:{}", ex);
            expLog("setNxEx()",ex);
            return null;
        }
    }

    public Boolean pSetNxEx(String key, String value, long milliseconds) {
        try {
            RedisCallback<String> callback = (connection) -> {
                JedisCommands commands = (JedisCommands) connection.getNativeConnection();
                return commands.set(key, value, "NX", "PX", milliseconds);
            };
            Object result = redisTemplate.execute(callback);
            if (result == null){
                return false;
            }
            else{
                return true;
            }
        } catch (Exception ex) {
            try {
                redisTemplate.delete(key);
            }
            catch (Exception e){
            }
            logger.error("pSetNxEx(),exception:{}", ex);
            expLog("pSetNxEx()",ex);
            return null;
        }
    }


    @Override
    public void hDel(String key, String... fields) {
        try {
            redisTemplate.opsForHash().delete(key, (Object[]) fields);
        } catch (Exception ex) {
            logger.error("hDel(),exception:{}", ex);
            expLog("hDel()",ex);
        }
    }

    @Override
    public Boolean hExists(String key, String field) {
        try {
            return redisTemplate.opsForHash().hasKey(key, field);
        } catch (Exception ex) {
            logger.error("hExists(),exception:{}", ex);
            expLog("hExists()",ex);
            return null;
        }
    }

    @Override
    public <V> V hGet(String key, String field) {
        try {
            V obj = (V) redisTemplate.opsForHash().get(key, field);
            return obj;
        } catch (Exception ex) {
            logger.error("hGet(),exception:{}", ex);
            expLog("hGet()",ex);
            return null;
        }
    }

    @Override
    public <V> Map<String, V> hGetAll(String key) {
        try {
            Map<String, V> map = redisTemplate.opsForHash().entries(key);
            return map;
        } catch (Exception ex) {
            logger.error("hGetAll(),exception:{}", ex);
            expLog("hGetAll()",ex);
            return null;
        }
    }

    @Override
    public Set<String> hKeys(String key) {
        try {
            return redisTemplate.opsForHash().keys(key);
        } catch (Exception ex) {
            logger.error("hKeys(),exception:{}", ex);
            expLog("hKeys()",ex);
            return null;
        }
    }

    @Override
    public <V> List<V> hMGet(String key, String... fields) {
        try {
            List<V> list = redisTemplate.opsForHash().multiGet(key, Arrays.asList(fields));
            return list;
        } catch (Exception ex) {
            logger.error("hMGet(),exception:{}", ex);
            expLog("hMGet()",ex);
            return null;
        }
    }

    @Override
    public <V> void hMSet(String key, Map<String, V> fieldMap) {
        try {
            redisTemplate.opsForHash().putAll(key, fieldMap);
        } catch (Exception ex) {
            logger.error("hMSet(),exception:{}", ex);
            expLog("hMSet()",ex);
        }
    }

    @Override
    public <V> void hSet(String key, String field, V value) {
        try {
            redisTemplate.opsForHash().put(key, field, value);
        } catch (Exception ex) {
            logger.error("hSet(),exception:{}", ex);
            expLog("hSet()",ex);
        }
    }

    @Override
    public <V> void hSetNx(String key, String field, V value) {
        try {
            redisTemplate.opsForHash().putIfAbsent(key, field, value);
        } catch (Exception ex) {
            logger.error("hSetNx(),exception:{}", ex);
            expLog("hSetNx()",ex);
        }
    }




//    @Override
//    public <V> List<Map<String, V>> pipHGetAll(String... keys) {
//        try {
//            List<Map<String, V>> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Map<String, V>>() {
//                @Override
//                public Map<String, V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.hGetAll(key.getBytes());
//                    }
//                    return null;
//                }
//            });
//            int fireCount = 0;
//            if (objects != null){
//                for(int i=0;i<objects.size();i++){
//                    Map<String, V> obj = objects.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return objects;
//        } catch (Exception ex) {
//            logger.error("pipHGetAll(),exception:{}", ex);
//            expLog("pipHGetAll()",ex);
//            return null;
//        }
//
//    }


//    @Override
//    public <V> void pipHMSet(Map<String,Map<String, V>> hMap) {
//        try {
//            Date now = new Date();
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Map<String,V>> e : hMap.entrySet()) {
//                        String key = e.getKey();
//                        Map<String,V> value = e.getValue();
//                        Map<byte[], byte[]> valMap = new HashMap<>();
//                        for (Map.Entry<String,V> ve:value.entrySet()){
//                            byte[] valBytes = cacheRedisTemplate.getHashValueSerializer().serialize(ve.getValue());
//                            valMap.put(ve.getKey().getBytes(),valBytes);
//                        }
//                        redisConnection.hMSet(key.getBytes(),(Map)valMap);
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipHGetAll(),exception:{}", ex);
//            expLog("pipHGetAll()",ex);
//        }
//
//    }

//    @Override
//    public <V> List<Map<String, V>> pipHashGetAll(String... keys) {
//        try {
//            List<String> objects = cacheRedisTemplate.executePipelined(new RedisCallback<String>() {
//                @Override
//                public String doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.get(key.getBytes());
//                    }
//                    return null;
//                }
//            }, new StringRedisSerializer());
//            List<Map<String, V>> results = null;
//            if (objects != null && objects.size() == keys.length){
//                results = new ArrayList<>();
//                for (int i=0;i<objects.size();i++){
//                    String object = objects.get(i);
//                    Map<String, V> map = new HashMap<>();
//                    if (StringUtils.isEmpty(object)){
//                        results.add(map);
//                    }
//                    else{
//                        try {
//                            //map = (Map<String, V>)com.alibaba.druid.support.json.JSONUtils.parse(object);
//                            map = (Map<String, V>)com.alibaba.fastjson.JSON.parseObject(object,Map.class);
//                        }
//                        catch (Exception ex){
//                            map = new HashMap<>();
//                        }
//                        results.add(map);
//                    }
//                }
//            }
//            int fireCount = 0;
//            if (results != null){
//                for(int i=0;i<results.size();i++){
//                    Map<String, V> obj = results.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return results;
//        } catch (Exception ex) {
//            logger.error("pipHashGetAll(),exception:{}", ex);
//            expLog("pipHashGetAll()",ex);
//            return null;
//        }
//
//    }

    @Override
    public <V> List<Map<String, V>> pipHashGetAll(String... keys) {
        try {
            List<Map<String, V>> objects = redisTemplate.executePipelined(new RedisCallback<Map<String, V>>() {
                @Override
                public Map<String, V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
                    for (String key : keys) {
                        redisConnection.get(key.getBytes());
                    }
                    return null;
                }
            });
//            int fireCount = 0;
//            if (objects != null && objects.size() == keys.length){
//                for(int i=0;i<objects.size();i++){
//                    Map<String, V> obj = objects.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
            return objects;
        } catch (Exception ex) {
            logger.error("pipHashGetAll(),exception:{}", ex);
            expLog("pipHashGetAll()",ex);
            return null;
        }

    }

//    @Override
//    public <V> void pipHashSetEx(Map<String,Map<String, V>> hMap, long seconds) {
//        try {
//            Date now = new Date();
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Map<String,V>> e : hMap.entrySet()) {
//                        String key = e.getKey();
//                        Map<String,V> value = e.getValue();
//                        //String val = com.alibaba.druid.support.json.JSONUtils.toJSONString(value);
//                        String val = com.alibaba.fastjson.JSON.toJSONString(value);
//                        if (val == null){
//                            val = "";
//                        }
//                        if (seconds > 0){
//                            redisConnection.setEx(key.getBytes(), seconds,val.getBytes());
//                        }
//                        else{
//                            redisConnection.set(key.getBytes(),val.getBytes());
//                        }
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipHashSetEx(),exception:{}", ex);
//            expLog("pipHashSetEx()",ex);
//        }
//
//    }

    @Override
    public <V> void pipHashSetEx(Map<String,Map<String, V>> hMap, long seconds) {
        try {
            Object object = redisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    for (Map.Entry<String,Map<String,V>> e : hMap.entrySet()) {
                        String key = e.getKey();
                        Map<String,V> value = e.getValue();
                        byte[] valBytes = redisTemplate.getValueSerializer().serialize(value);
                        if (seconds > 0){
                            redisConnection.setEx(key.getBytes(), seconds,valBytes);
                        }
                        else{
                            redisConnection.set(key.getBytes(),valBytes);
                        }
                    }
                    return null;
                }
            });
        } catch (Exception ex) {
            logger.error("pipHashSetEx(),exception:{}", ex);
            expLog("pipHashSetEx()",ex);
        }

    }

    @Override
    public <V> void sAdd(String key, V... members) {
        try {
            redisTemplate.opsForSet().add(key, members);
        } catch (Exception ex) {
            logger.error("sAdd(),exception:{}", ex);
            expLog("sAdd()",ex);
        }
    }

    @Override
    public Long sCard(String key) {
        try {
            return redisTemplate.opsForSet().size(key);
        } catch (Exception ex) {
            logger.error("sCard(),exception:{}", ex);
            expLog("sCard()",ex);
            return null;
        }
    }

    @Override
    public Boolean sIsMember(String key, Object member) {
        try {
            return redisTemplate.opsForSet().isMember(key, member);
        } catch (Exception ex) {
            logger.error("sIsMember(),exception:{}", ex);
            expLog("sIsMember()",ex);
            return null;
        }
    }

    @Override
    public <V> Set<V> sMembers(String key) {
        try {
            return redisTemplate.opsForSet().members(key);
        } catch (Exception ex) {
            logger.error("sMembers(),exception:{}", ex);
            expLog("sMembers()",ex);
            return null;
        }
    }

//    @Override
//    public <V> List<Set<V>> pipSMembers(String... keys) {
//        try {
//            List<Set<V>> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Set<V>>() {
//                @Override
//                public Set<V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.sMembers(key.getBytes());
//                    }
//                    return null;
//                }
//            });
//            int fireCount = 0;
//            if (objects != null){
//                for(int i=0;i<objects.size();i++){
//                    Set<V> obj = objects.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return objects;
//        } catch (Exception ex) {
//            logger.error("pipSMembers(),exception:{}", ex);
//            expLog("pipSMembers()",ex);
//            return null;
//        }
//
//    }

//    @Override
//    public <V> void pipSAdd(Map<String,Set<V>> sMap) {
//        try {
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Set<V>> e : sMap.entrySet()) {
//                        String key = e.getKey();
//                        Set<V> value = e.getValue();
//                        List<byte[]> vals = new ArrayList<>();
//                        for (V ve:value){
//                            byte[] valBytes = cacheRedisTemplate.getHashValueSerializer().serialize(ve);
//                            vals.add(valBytes);
//                        }
//                        //byte[][] valArray = (byte[][])vals.toArray();
//                        byte[][] valArray = vals.toArray(new byte[0][0]);
//                        redisConnection.sAdd(key.getBytes(),valArray);
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipSAdd(),exception:{}", ex);
//            expLog("pipSAdd()",ex);
//        }
//    }

//    @Override
//    public <V> List<Set<V>> pipSetMembers(String... keys) {
//        try {
//            List<String> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Set<V>>() {
//                @Override
//                public Set<V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.get(key.getBytes());
//                    }
//                    return null;
//                }
//            },new StringRedisSerializer());
//            List<Set<V>> results = null;
//            if (objects != null && objects.size() == keys.length){
//                results = new ArrayList<>();
//                for (int i=0;i<objects.size();i++){
//                    String object = objects.get(i);
//                    Set<V> set = new HashSet<>();
//                    if (StringUtils.isEmpty(object)){
//                        results.add(set);
//                    }
//                    else{
//                        try {
//                            //List list = (List)com.alibaba.druid.support.json.JSONUtils.parse(object);
//                            List list = (List) com.alibaba.fastjson.JSON.parseObject(object,List.class);
//                            set = new HashSet<>(list);
//                        }
//                        catch (Exception ex){
//                            set = new HashSet<>();
//                        }
//                        results.add(set);
//                    }
//                }
//            }
//            int fireCount = 0;
//            if (results != null){
//                for(int i=0;i<results.size();i++){
//                    Set<V> obj = results.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return results;
//        } catch (Exception ex) {
//            logger.error("pipSMembers(),exception:{}", ex);
//            expLog("pipSMembers()",ex);
//            return null;
//        }
//
//    }

    @Override
    public <V> List<Set<V>> pipSetMembers(String... keys) {
        try {
            List<Set<V>> objects = redisTemplate.executePipelined(new RedisCallback<Set<V>>() {
                @Override
                public Set<V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
                    for (String key : keys) {
                        redisConnection.get(key.getBytes());
                    }
                    return null;
                }
            });

//            int fireCount = 0;
//            if (objects != null && objects.size() == keys.length){
//                for(int i=0;i<objects.size();i++){
//                    Set<V> obj = objects.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
            return objects;
        } catch (Exception ex) {
            logger.error("pipSMembers(),exception:{}", ex);
            expLog("pipSMembers()",ex);
            return null;
        }
    }

    @Override
    public <V> void pipSetAddEx(Map<String,Set<V>> sMap, long seconds) {
        try {
            Object object = redisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    for (Map.Entry<String,Set<V>> e : sMap.entrySet()) {
                        String key = e.getKey();
                        Set<V> value = e.getValue();
                        byte[] valBytes = redisTemplate.getValueSerializer().serialize(value);
                        if (seconds > 0){
                            redisConnection.setEx(key.getBytes(), seconds,valBytes);
                        }
                        else{
                            redisConnection.set(key.getBytes(),valBytes);
                        }
                    }
                    return null;
                }
            });
        } catch (Exception ex) {
            logger.error("pipSAdd(),exception:{}", ex);
            expLog("pipSAdd()",ex);
        }
    }

//    @Override
//    public <V> void pipSetAddEx(Map<String,Set<V>> sMap, long seconds) {
//        try {
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Set<V>> e : sMap.entrySet()) {
//                        String key = e.getKey();
//                        Set<V> value = e.getValue();
//                        //String val = com.alibaba.druid.support.json.JSONUtils.toJSONString(value);
//                        String val = com.alibaba.fastjson.JSON.toJSONString(value);
//                        if (val == null){
//                            val = "";
//                        }
//                        if (seconds > 0){
//                            redisConnection.setEx(key.getBytes(), seconds,val.getBytes());
//                        }
//                        else{
//                            redisConnection.set(key.getBytes(),val.getBytes());
//                        }
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipSAdd(),exception:{}", ex);
//            expLog("pipSAdd()",ex);
//        }
//    }

    @Override
    public void sRem(String key, String... members) {
        try {
            redisTemplate.opsForSet().remove(key, (Object[])members);
        } catch (Exception ex) {
            logger.error("sRem(),exception:{}", ex);
            expLog("sRem()",ex);
        }
    }

    // ===============================sorted set start=================================

    /**
     * 向有序集合添加一个成员的
     *
     * ZADD key score1 member1 [score2 member2]
     *
     */
    public boolean zAdd(String key, Object member, double score, int time) {
        try {
            redisTemplate.opsForZSet().add(key, member, score);
            if (time > 0)
                expire(key, time);
            return true;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return false;
        }
    }
    /**
     * 向有序集合添加多个成员的
     *
     * Add {@code tuples} to a sorted set at {@code key}, or update its {@code score}
     * @param key must not be {@literal null}.
     * @param tuples must not be {@literal null}.
     * @return
     * @see <a href="http://redis.io/commands/zadd">Redis Documentation: ZADD</a>
     *
     */
    public boolean zAdds(String key, Set<ZSetOperations.TypedTuple<Object>> tuples, int time) {
        try {
            redisTemplate.opsForZSet().add(key,tuples);
            if (time > 0)
                expire(key, time);
            return true;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return false;
        }
    }
    /**
     * 向有序集合添加一个成员,范围指定
     *
     * ZADD key score1 member1 [score2 member2]
     *
     */
    public boolean zAddWithinCount(String key, Object member, double score, int time, long count) {
        try {
            redisTemplate.opsForZSet().add(key, member, score);
            if (time > 0)
                expire(key, time);
            Long redisTotalCount = redisTemplate.opsForZSet().zCard(key);
            if (redisTotalCount > count){
                redisTemplate.opsForZSet().removeRange(key, count, -1);
            }
            return true;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return false;
        }
    }
    /**
     * 	ZRANGEBYSCORE key min max [WITHSCORES] [LIMIT]
     通过分数返回有序集合指定区间内的成员
     *
     */
    public Set<Object> zRangeByScore(String key, double minScore, double maxScore) {
        try {
            return redisTemplate.opsForZSet().rangeByScore(key, minScore, maxScore);
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }

    /**
     * 	ZRANGEBYSCORE key min max [WITHSCORES] [LIMIT]
     通过分数返回有序集合指定区间内的成员
     *
     */
    public Set<Object> zReverseRangeByscore(String key, double minScore, double maxScore) {
        try {
            return redisTemplate.opsForZSet().reverseRangeByScore(key, minScore, maxScore);
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }

    /**
     * 	ZRANGEBYSCORE key min max [WITHSCORES] [LIMIT]
     通过分数返回有序集合指定区间内的成员
     *
     */
    public Set<Object> zReverseRangeByscore(String key, double minScore, double maxScore, long offset, long count) {
        try {
            return redisTemplate.opsForZSet().reverseRangeByScore(key, minScore, maxScore, offset, count);
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }

    /**
     * 	ZSCORE key member
     返回有序集中，成员的分数值
     *
     */
    public Double zScore(String key, Object member) {
        try {
            return redisTemplate.opsForZSet().score(key, member);
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }

    /**
     * 	ZRANK key member 返回有序集合中指定成员的索引
     *
     */
    public Long zRank(String key, Object member) {
        try {
            return redisTemplate.opsForZSet().rank(key, member);
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }

    /**
     * 	Get the size of sorted set with {@code key}.
     *
     */
    public Long zCard(String key) {
        try {
            return redisTemplate.opsForZSet().zCard(key);
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }
    /**
     * Zscan 迭代有序集合中的元素（包括元素成员和元素分值）
     *
     */
    public Cursor<ZSetOperations.TypedTuple<Object>> zscan(String key) {
        try {
            Cursor<ZSetOperations.TypedTuple<Object>> cursor = redisTemplate.opsForZSet().scan(key, ScanOptions.NONE);
            return cursor;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return null;
        }
    }


    /**
     * Remove {@code values} from sorted set. Return number of removed elements.
     *
     * @param key must not be {@literal null}.
     * @param values must not be {@literal null}.
     * @return
     * @see <a href="http://redis.io/commands/zrem">Redis Documentation: ZREM</a>
     */
    public boolean zRem(String key, Object... values) {
        try {
            redisTemplate.opsForZSet().remove(key, values);
            return true;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return false;
        }
    }

    /**
     * Remove elements with scores between {@code min} and {@code max} from sorted set with {@code key}.
     *
     * @param key must not be {@literal null}.
     * @param min
     * @param max
     * @return
     * @see <a href="http://redis.io/commands/zremrangebyscore">Redis Documentation: ZREMRANGEBYSCORE</a>
     */
    public boolean zRemrangebyscore (String key, double min, double max) {
        try {
            redisTemplate.opsForZSet().removeRangeByScore (key, min, max);
            return true;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return false;
        }
    }

    /**
     * Remove elements in range between {@code start} and {@code end} from sorted set with {@code key}.
     *
     * @param key must not be {@literal null}.
     * @param start
     * @param end
     * @return
     * @see <a href="http://redis.io/commands/zremrangebyrank">Redis Documentation: ZREMRANGEBYRANK</a>
     */
    public boolean zRemoveRange(String key, long start, long end) {
        try {
            redisTemplate.opsForZSet().removeRange(key, start, end);
            return true;
        } catch (Exception e) {
            logger.error("redis error: ", e);
            return false;
        }
    }
    // ===============================sorted set end=================================


    @Override
    public void flushdb() {
        try {
            redisTemplate.execute(new RedisCallback<Object>() {
                public String doInRedis(RedisConnection connection) throws DataAccessException {
                    connection.flushDb();
                    return null;
                }
            });
        } catch (Exception ex) {
            logger.error("flushdb(),exception:{}", ex);
            expLog("flushdb()",ex);
        }
    }
}
