package com.valor.mfc.vms.common.database.tool.configruation;

import com.google.common.base.Strings;
import common.config.tools.config.ConfigAESTools;
import common.config.tools.config.ConfigTools3;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;

import javax.sql.DataSource;
import java.util.Properties;

import static java.sql.Connection.TRANSACTION_READ_COMMITTED;

/**
 * Created by Frank.Huang on 2016/4/23.
 */
public abstract class AbstractDBCfg {
    private static final Logger logger = LoggerFactory.getLogger(AbstractDBCfg.class);

    protected String driverClassName = ConfigTools3.getString("db.driverClassName", "com.mysql.jdbc.Driver");
    protected String dialect = ConfigTools3.getString("db.dialect", "org.hibernate.dialect.MySQL5Dialect");
    protected String validationQuery = ConfigTools3.getString("db.validationQuery", "SELECT 1");
    protected String hbm2ddlAuto = ConfigTools3.getString("db.hbm2ddlAuto", "update");

    protected boolean showSql = ConfigTools3.getBoolean("db.stat.showSql", false);
    protected boolean generateStatistics = ConfigTools3.getBoolean("db.stat.generateStatistics", false);

    protected boolean useL2Cache = ConfigTools3.getBoolean("db.l2cache.useL2Cache", false);
    protected boolean useQueryCache = ConfigTools3.getBoolean("db.l2cache.pool.useQueryCache", false);
    protected String l2CacheFactory = ConfigTools3.getString("db.l2cache.l2CacheFactory", "org.hibernate.cache.ehcache.EhCacheRegionFactory");

    public abstract DataSource getDataSource(String dbURL, String username, String password);

    public Properties getDBProperties() {
        Properties props = new Properties();

        props.put("hibernate.dialect", dialect);
        props.put("hibernate.show_sql", showSql);
        props.put("hibernate.format_sql", false);
        props.put("hibernate.generate_statistics", generateStatistics);
        props.put("hibernate.hbm2ddl.auto", hbm2ddlAuto);
        props.put("hibernate.connection.isolation", TRANSACTION_READ_COMMITTED);
        props.put("hibernate.use_sql_comments", true);

        props.put("hibernate.cache.use_second_level_cache", false);
        props.put("hibernate.cache.use_query_cache", false);
        props.put("hibernate.connection.CharSet", "utf8");
        props.put("hibernate.connection.characterEncoding", "utf8");
        props.put("hibernate.connection.useUnicode", true);
        props.put("hibernate.autoReconnect", true);
        return props;
    }

    public FactoryBean<SessionFactory> getSessionFactory(
        DataSource dataSource, String... scanPackages) {
        Properties props = getDBProperties();
        LocalSessionFactoryBean sessionFactoryBean = new LocalSessionFactoryBean();
        sessionFactoryBean.setEntityInterceptor(new AuditInterceptor());
        sessionFactoryBean.setDataSource(dataSource);
        sessionFactoryBean.setPackagesToScan(scanPackages);
        sessionFactoryBean.setHibernateProperties(getDBProperties());
        // sessionFactoryBean.setNamingStrategy();

        return sessionFactoryBean;
    }

    public PlatformTransactionManager getTransactionManager(
        SessionFactory sessionFactory, int txTimeout) throws Exception {
        HibernateTransactionManager transactionManager =
            new HibernateTransactionManager(sessionFactory);
        transactionManager.setDefaultTimeout(txTimeout);
        return transactionManager;
    }

    public String getConnectUrl(String host, String inst) {
        if (Strings.isNullOrEmpty(host) || Strings.isNullOrEmpty(inst)) {
            return "";
        }

        return String.format("jdbc:mysql://%s/%s?useUnicode=true&characterEncoding=utf8&autoReconnect=true", host, inst);
    }

    /**
     * get database config parameter
     * if crypt key is Configured decrypt parameter use crypt key,otherwise return value direct.
     *
     * @param configKey
     * @param cryptKey
     * @return
     */
    protected String getDBConfig(String configKey, String cryptKey) {

        return Strings.isNullOrEmpty(cryptKey) ? ConfigTools3.getString(configKey) : ConfigAESTools.getAESString(configKey, cryptKey);

    }
}
