package com.valor.mfc.vms.common.database.tool.configruation;


import org.apache.commons.dbcp2.BasicDataSource;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;

import javax.sql.DataSource;
import java.util.Properties;

import static java.sql.Connection.TRANSACTION_READ_COMMITTED;

/**
 * Created by Frank.Huang on 2016/4/23.
 */
//@Configuration
public abstract class AbstractDBConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(AbstractDBConfiguration.class);

    @Value("${vms.database.driver:com.mysql.jdbc.Driver}")
    protected String driverClassName;

    @Value("${vms.database.dialect:org.hibernate.dialect.MySQL5Dialect}")
    protected String dialect;

    @Value("${vms.database.poolPreparedStatements:true}")
    protected boolean poolPreparedStatements;

    @Value("${vms.database.validationQuery:SELECT 1}")
    protected String validationQuery;

    @Value("${vms.database.connections.initialSize:5}")
    private int initialSize;

    @Value("${vms.database.connections.maxActive:500}")
    private int maxActive;

    @Value("${vms.database.connections.minIdle:2}")
    private int minIdle;

    @Value("${vms.database.connections.maxIdle:10}")
    protected int maxIdle;

    @Value("${vms.database.connections.maxWait:500}")
    protected long maxWait;

    @Value("${vms.database.connections.minEvictableIdleTimeMillis:1800000}")
    protected long minEvictableIdleTimeMillis;

    @Value("${vms.database.connections.timeBetweenEvictionRunsMillis:1800000}")
    protected long timeBetweenEvictionRunsMillis;

    @Value("${vms.database.connections.numTestsPerEvictionRun:3}")
    protected int numTestsPerEvictionRun;

    @Value("${vms.database.connections.testOnBorrow:true}")
    protected boolean testOnBorrow;

    @Value("${vms.database.connections.testWhileIdle:true}")
    protected boolean testWhileIdle;

    @Value("${vms.database.connections.testOnReturn:true}")
    protected boolean testOnReturn;

    @Value("${vms.database.useL2Cache:false}")
    protected boolean useL2Cache;

    @Value("${vms.database.useQueryCache:false}")
    protected boolean useQueryCache;

    @Value("${vms.database.l2CacheFactory:org.hibernate.cache.ehcache.EhCacheRegionFactory}")
    protected String l2CacheFactory;

    @Value("${vms.database.showSql:false}")
    protected boolean showSql;

    @Value("${vms.database.statistics:false}")
    protected boolean generateStatistics;

    @Value("${vms.database.auto:update}")
    protected String hbm2ddlAuto;


    public DataSource getDataSource(String dburl, String username, String password) {
        BasicDataSource datasource = new BasicDataSource();
        datasource.setUrl(dburl);
        datasource.setUsername(username);
        datasource.setPassword(password);

        datasource.setDriverClassName(driverClassName);
        datasource.setValidationQuery(validationQuery);
        datasource.setPoolPreparedStatements(poolPreparedStatements);
        datasource.setInitialSize(initialSize);
        datasource.setMaxTotal(maxActive);
        datasource.setMinIdle(minIdle);
        datasource.setMaxIdle(maxIdle);
        datasource.setMaxWaitMillis(maxWait);
        datasource.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);
        datasource.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        datasource.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        datasource.setLogAbandoned(true);
        datasource.setRemoveAbandonedOnBorrow(true);
        datasource.setRemoveAbandonedOnMaintenance(true);
        datasource.setRemoveAbandonedTimeout(180);
        datasource.setTestOnBorrow(testOnBorrow);
        datasource.setTestWhileIdle(testWhileIdle);
        datasource.setTestOnReturn(testOnReturn);
        datasource.setDefaultTransactionIsolation(TRANSACTION_READ_COMMITTED);
        datasource.setValidationQuery("select 1");

        logger.info("DataSource:{}", datasource.toString());
        return datasource;
    }

    public Properties getDBProperties(){
        Properties props = new Properties();

        props.put("hibernate.dialect", dialect);
        props.put("hibernate.show_sql", showSql);
        props.put("hibernate.format_sql",false);
        props.put("hibernate.generate_statistics", generateStatistics);
        props.put("hibernate.hbm2ddl.auto", "none");
        props.put("hibernate.connection.isolation", TRANSACTION_READ_COMMITTED);
        props.put("hibernate.use_sql_comments", true);

        props.put("hibernate.cache.use_second_level_cache", false);
        props.put("hibernate.cache.use_query_cache", false);
        props.put("hibernate.connection.CharSet", "utf8");
        props.put("hibernate.connection.characterEncoding", "utf8");
        props.put("hibernate.connection.useUnicode", true);
        props.put("hibernate.autoReconnect", true);

        //props.put("hibernate.current_session_context_class","thread");
        //props.put("hibernate.current_session_context_class","org.springframework.orm.hibernate4.SpringSessionContext");


        return props;
    }

    public FactoryBean<SessionFactory> getSessionFactory(DataSource dataSource,String...scanPackages) {
        Properties props = getDBProperties();
        LocalSessionFactoryBean sessionFactoryBean = new LocalSessionFactoryBean();
        sessionFactoryBean.setEntityInterceptor(new AuditInterceptor());
        sessionFactoryBean.setDataSource(dataSource);
        sessionFactoryBean.setPackagesToScan(scanPackages);
        sessionFactoryBean.setHibernateProperties(getDBProperties());
        //sessionFactoryBean.setNamingStrategy();

        return sessionFactoryBean;
    }

    public PlatformTransactionManager getTransactionManager(SessionFactory sessionFactory,int txTimeout) throws Exception {
        HibernateTransactionManager transactionManager = new HibernateTransactionManager(sessionFactory);
        transactionManager.setDefaultTimeout(txTimeout);
        return transactionManager;
    }

}
