package com.valor.mfc.vms.common.reg;

import com.google.common.base.Strings;
import com.valor.mfc.vms.common.cache.redis.RedisCache;
import common.base.tools.sys.SystemTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;


public class ServiceRegister {
    private static final Logger logger = LoggerFactory.getLogger(ServiceRegister.class);
    private RedisCache redisCache;
    private RegisterServiceEntity localService = null;
    private ScheduledExecutorService refreshExecutor = Executors.newScheduledThreadPool(2);
    private long refreshSeconds = 300;//5分钟
    private long timeoutSeconds = 720;//12分钟

    public ServiceRegister(RedisCache redisCache){
        this.redisCache =  redisCache;
        refreshExecutor.scheduleAtFixedRate(new Runnable() {
            @Override
            public void run() {
                try {
                    refresh();
                }
                catch (Exception ex){}
            }
        }, 0, refreshSeconds, TimeUnit.SECONDS);
    }


    private String  getRedisKey(String serviceName){
        return "services-"+serviceName;
    }

    private void refresh(){
        if (localService == null){
            logger.info("refresh(),no local service to refresh");
            return;
        }
        logger.info("refresh(),serviceName:{}",localService.getServiceName());
        String redisKey = getRedisKey(localService.getServiceName());
        List<RegisterServiceEntity> services = redisCache.getWithException(redisKey);
        if (services == null){services = new ArrayList<>();}
        if (services.size() == 0){
            logger.info("refresh(),no service is found.serviceName:{}",localService.getServiceName());
        }
        else{
            logger.info("refresh(),{} service is found.serviceName:{},ip:{}",services.size(),localService.getServiceName(),printIp(services));
        }


        List<RegisterServiceEntity> newServices = new ArrayList<>();
        boolean containLocal = false;
        for (int i=0;i<services.size();i++){
            RegisterServiceEntity ser = services.get(i);
            if (ser.getIp().equals(localService.getIp())){
                ser.setCreateTime(System.currentTimeMillis());
                newServices.add(ser);
                containLocal = true;
            }
            else{
                long lastTime = ser.getCreateTime();
                if (System.currentTimeMillis() - lastTime < timeoutSeconds * 1000){
                    newServices.add(ser);
                }
                else{
                    logger.info("refresh(),ip {} is timeout,remove it.lastTime:{}",ser.getIp(),ser.getCreateTime());
                }
            }
        }
        if (!containLocal){
            RegisterServiceEntity cloneSer = new RegisterServiceEntity();
            cloneSer.setIp(localService.getIp());
            cloneSer.setServiceName(localService.getServiceName());
            cloneSer.setCreateTime(System.currentTimeMillis());
            newServices.add(cloneSer);
        }
        logger.info("refresh(),{} service is refreshed.serviceName:{},ip:{}",newServices.size(),localService.getServiceName(),printIp(newServices));
        redisCache.set(redisKey,newServices);
    }

    private String printIp(List<RegisterServiceEntity> entities){
        StringBuilder ipSb = new StringBuilder();
        for (int i=0;i<entities.size();i++){
            if (i != 0){
                ipSb.append(",");
            }
            ipSb.append(entities.get(i).getIp());
        }
        return ipSb.toString();
    }

    public void register(String serviceName,String ethernet,long refreshSeconds,long timeoutSeconds){
        logger.info("register(),serviceName:{},ethernet:{}",serviceName,ethernet);
        this.refreshSeconds = refreshSeconds;
        this.timeoutSeconds = timeoutSeconds;
        //获取IP地址
        String ipAddr = SystemTools.getIpAddress(true,ethernet);
        if (Strings.isNullOrEmpty(ipAddr)){
            return;
        }
        RegisterServiceEntity ser = new RegisterServiceEntity();
        ser.setIp(ipAddr);
        ser.setServiceName(serviceName);
        ser.setCreateTime(System.currentTimeMillis());
        localService = ser;
        //refresh();
        logger.info("register(),service is registered.serviceName:{},ethernet:{},ip:{}",serviceName,ethernet,ipAddr);
    }

    public List<RegisterServiceEntity> getServices(String serviceName){
        logger.info("getServices(),serviceName:{}",serviceName);
        String redisKey = getRedisKey(serviceName);
        List<RegisterServiceEntity> services = redisCache.get(redisKey);
        if (services == null || services.size() == 0){
            logger.info("getServices(),no service is found.serviceName:{}",serviceName);
            return new ArrayList<>();
        }
        logger.info("getServices(),{} service is found.serviceName:{},ip:{}",services.size(),serviceName,printIp(services));
        return services;
    }

}
