package com.valor.mfc.vms.common.tools.meta;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import common.base.tools.type.CollectionUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

/**
 * Created by Frank.Huang on 2016/7/27.
 */
public class LanguageTools {
    private static final Logger logger = LoggerFactory.getLogger(LanguageTools.class);

    public static String LANG_PT = "pt";
    public static String LANG_ES = "es";
    public static String LANG_EN = "en";
    public static String LANG_ES_TMDB_ALIAS = "spa";

    public static boolean isPortuguese(String language){
        return LANG_PT.equalsIgnoreCase(language);
    }

    public static boolean isSpanish(String language){
        return LANG_ES.equalsIgnoreCase(language);
    }

    public static boolean isEnglish(String language){
        return LANG_EN.equalsIgnoreCase(language);
    }

    public static boolean isSupportLanguage(String language){
        return isEnglish(language)||isPortuguese(language)||isSpanish(language);
    }

    public static String spanishAlias(String language){
        if (LANG_ES_TMDB_ALIAS.equalsIgnoreCase(language)){
            return LANG_ES;
        }else {
            return language;
        }
    }

    public static boolean equalLanguage(String lang1,String lang2){
        if (Strings.isNullOrEmpty(lang1) || Strings.isNullOrEmpty(lang2)){
            return false;
        }

        if (lang1.equalsIgnoreCase(lang2)){
            return true;
        }

        if (isEnglish(lang1)|| isEnglish(lang2)){
            if (spanishAlias(lang1).equalsIgnoreCase(spanishAlias(lang2))){
                return true;
            }
        }

        return false;
    }

    public static String getLanguage(String languageRegion){
        List<String> langs = getLanguageRegionList(languageRegion);
        if (langs.size() > 1){
            return langs.get(1);
        }else {
            return "";
        }
    }

    public static String getLanguageRegion(String languageRegion){
        List<String> langs = getLanguageRegionList(languageRegion);
        if (langs.size() > 0){
            return langs.get(0);
        }else {
            return "";
        }
    }

    public static List<String> getLanguageRegionList(String languageRegion) {
        try {
            return Splitter.on("_")
                    .omitEmptyStrings().trimResults()
                    .splitToList(languageRegion);

        } catch (Exception e) {
            logger.info("Invalid language args:{}", languageRegion);
            return Lists.newArrayList();
        }
    }

    public static String getOrDefaultLanguage(String language){
        if (isSupportLanguage(language)){
            return language;
        }else {
            return LanguageTools.LANG_EN;
        }
    }

    public static boolean isContainEnglish(String originalLanguage) {
        if (Strings.isNullOrEmpty(originalLanguage)) {
            return false;
        }

        try {
            Map<String, String> languageMap = Splitter.on("|")
                    .omitEmptyStrings()
                    .trimResults().withKeyValueSeparator(":")
                    .split(originalLanguage);

            if (!CollectionUtils.isNullOrEmpty(languageMap)) {
                if (languageMap.containsKey(LANG_EN)) {
                    return true;
                }
            }
        } catch (IllegalArgumentException e) {
            logger.warn("{}:{}", originalLanguage, e.getMessage());
        }

        return false;
    }
}
