/*
 *      Copyright (c) 2004-2015 YAMJ Members
 *      https://github.com/organizations/YAMJ/teams
 *
 *      This file is part of the Yet Another Media Jukebox (YAMJ).
 *
 *      YAMJ is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, either version 3 of the License, or
 *      any later version.
 *
 *      YAMJ is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with YAMJ.  If not, see <http://www.gnu.org/licenses/>.
 *
 *      Web: https://github.com/YAMJ/yamj-v3
 *
 */
package com.valor.mfc.vms.common.tools.type;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class DateTimeTools {
    private static final Logger logger = LoggerFactory.getLogger(DateTimeTools.class);

    private static final String DATE_FORMAT_STRING = "yyyy-MM-dd";
    private static final PeriodFormatter TIME_FORMAT_COLON = createPeriodFormatter(":", ":", "");
    private static final PeriodFormatter TIME_FORMAT_TEXT = createPeriodFormatter("h", "m", "s");
    // Some default formats in use
    public static final String ISO8601_FORMAT = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    public static final String BUILD_FORMAT = "yyyy-MM-dd HH:mm:ss Z";

    public static final String DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

    public static final String VMS_TIMEZONE ="GMT-3"; //设置系统默认时区为GMT-3 (巴西/BR）。
//    public static final String VMS_TIMEZONE ="GMT+8"; //设置系统默认时区为GMT-3 (巴西/BR）。

    public static final long SECONDS_OF_DAY = 60*60*24;  //每天的秒数
    public static final long MS_OF_DAY = SECONDS_OF_DAY*1000;  //每天的秒数

    private DateTimeTools() {
        throw new UnsupportedOperationException("Utility class");
    }

    /**
     * Create a Period Formatter with the given delimiters
     *
     * @param hourText
     * @param minuteText
     * @param secondText
     * @return
     */
    private static PeriodFormatter createPeriodFormatter(String hourText, String minuteText, String secondText) {
        return new PeriodFormatterBuilder()
                .appendHours()
                .appendSeparator(hourText)
                .minimumPrintedDigits(2)
                .appendMinutes()
                .appendSeparator(minuteText)
                .appendSecondsWithOptionalMillis()
                .appendSuffix(secondText)
                .toFormatter();
    }

    /**
     * Convert a date to a string using the DATE_FORMAT
     *
     * @param convertDate
     * @return converted date in the format specified in DATE_FORMAT_STRING
     */
    public static String convertDateToString(Date convertDate) {
        return convertDateToString(convertDate, DATE_FORMAT_STRING);
    }

    public static String datetimeToString(Date convertDate) {
        return convertDateToString(convertDate, DATE_TIME_FORMAT);
    }

    /**
     * Convert a date to a string using the supplied format
     *
     * @param convertDate
     * @param dateFormat
     * @return
     */
    public static String convertDateToString(Date convertDate, final String dateFormat) {
        DateTime dt = new DateTime(convertDate);
        return convertDateToString(dt, dateFormat);
    }

    /**
     * Convert a date to a string using the DATE_FORMAT
     *
     * @param convertDate
     * @return converted date in the format specified in DATE_FORMAT_STRING
     */
    public static String convertDateToString(DateTime convertDate) {
        return convertDateToString(convertDate, DATE_FORMAT_STRING);
    }

    /**
     * Convert a date to a string using the supplied format
     *
     * @param convertDate
     * @param dateFormat
     * @return
     */
    public static String convertDateToString(DateTime convertDate, final String dateFormat) {
        DateTimeFormatter fmt = DateTimeFormat.forPattern(dateFormat);
        return fmt.print(convertDate);
    }

    /**
     * Get the duration between two Java Dates
     *
     * @param start
     * @param end
     * @return
     */
    public static long getDuration(Date start, Date end) {
        return getDuration(new DateTime(start), new DateTime(end));
    }

    /**
     * Get the duration between two Long Dates
     *
     * @param start
     * @param end
     * @return
     */
    public static long getDuration(Long start, Long end) {
        return getDuration(new DateTime(start), new DateTime(end));
    }

    /**
     * Get the duration between tow Joda Dates
     *
     * @param start
     * @param end
     * @return the difference (in milliseconds) or -1 if "start" is after "end"
     */
    public static long getDuration(DateTime start, DateTime end) {
        if (start.isBefore(end)) {
            Interval interval = new Interval(start, end);
            return interval.toDurationMillis();
        }
        return -1L;
    }

    /**
     * Format the duration in milliseconds as ?:?:? format
     *
     * @param milliseconds
     * @return
     */
    public static String formatDurationColon(long milliseconds) {
        return formatDuration(milliseconds, TIME_FORMAT_COLON);
    }

    /**
     * Format the duration in milliseconds as ?h?m?s format
     *
     * @param milliseconds
     * @return
     */
    public static String formatDurationText(long milliseconds) {
        return formatDuration(milliseconds, TIME_FORMAT_TEXT);
    }

    /**
     * Format the duration in milliseconds in the given format
     *
     * @param milliseconds
     * @param format
     * @return
     */
    public static String formatDuration(long milliseconds, PeriodFormatter format) {
        Period period = new Period(milliseconds, PeriodType.time());
        period = period.normalizedStandard();
        return format.print(period);
    }

    /**
     * Take a string runtime in various formats and try to output this in minutes
     *
     * @param runtime
     * @return
     */
    public static int processRuntime(String runtime) {
        int returnValue;
        // See if we can convert this to a number and assume it's correct if we can
        try {
            returnValue = Integer.parseInt(runtime);
            return returnValue;
        } catch (Exception ignore) {
            returnValue = -1;
        }

        // This is for the format xx(hour/hr/min)yy(min), e.g. 1h30, 90mins, 1h30m
        Pattern hrmnPattern = Pattern.compile("(?i)(\\d+)(\\D*)(\\d*)(.*?)");

        Matcher matcher = hrmnPattern.matcher(runtime);
        if (matcher.find()) {
            String first = matcher.group(1);
            String divide = matcher.group(2);
            String second = matcher.group(3);

            if (StringUtils.isNotBlank(second)) {
                // Assume that this is HH(text)MM
                returnValue = (Integer.parseInt(first) * 60) + Integer.parseInt(second);
                return returnValue;
            }

            if (StringUtils.isBlank(divide)) {
                // No divider value, so assume this is a straight minute value
                returnValue = Integer.parseInt(first);
                return returnValue;
            }

            if (StringUtils.isBlank(second) && StringUtils.isNotBlank(divide)) {
                // this is xx(text) so we need to work out what the (text) is
                if (divide.toLowerCase().contains("h")) {
                    // Assume it is a form of "hours", so convert to minutes
                    returnValue = Integer.parseInt(first) * 60;
                } else {
                    // Assume it's a form of "minutes"
                    returnValue = Integer.parseInt(first);
                }
                return returnValue;
            }
        }

        return returnValue;
    }

    /**
     * Parses a string in the provided format to a DateTime
     *
     * @param stringDate the date to parse
     * @param datePattern the pattern to parse
     * @return
     */
    public static DateTime parseDate(String stringDate, String datePattern) {
        DateTimeFormatter formatter = DateTimeFormat.forPattern(datePattern);
        return formatter.parseDateTime(stringDate);
    }

    public static long parseDate2Long(String stringDate, String datePattern){
        DateTime dt = parseDate(stringDate,datePattern);
        if (dt!= null){
            return dt.getMillis();
        }else {
            return 0L;
        }
    }

    /**
     * 获取据当前的时间间隔。
     * @param diff
     * @return
     */
    public static String distanceToday(int diff){
        long date = diff * (long)(24*60*60*1000);
        Date diffDate = new Date(System.currentTimeMillis() + date);

        return getDateStr(diffDate,"yyyy-MM-dd");

    }

    public static Date distanceTodayDate(int diff){
        long date = diff * (long)(24*60*60*1000);
        Date diffDate = new Date(System.currentTimeMillis() + date);
        return diffDate;
    }


    public static int distanceTodayInt(int diff){
        return date2IntVal(distanceToday(diff));
    }

    public static long distanceTodayMS(int diff){
        long distanceMS = (long)diff *1000*60*60*24;
        return System.currentTimeMillis() + distanceMS;
    }


    public static Date str2Date(String dateStr){
        try {
            return  new SimpleDateFormat("yyyy-MM-dd").parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 转换String日期为int类型,转换后数据如下：
     *      yyyy-mm-dd  -> yyyymmdd
     *      yyyy-mm     -> yyyymm00
     *      yyyy        -> yyyy0000
     * @param strDate
     * @return
     */
    public static int date2IntVal(String strDate){
        if (Strings.isNullOrEmpty(strDate)){
            return 0;
        }

        List<String> splitDate = Lists.newArrayList();

        try {
            splitDate = Splitter.on("-").trimResults().limit(3).splitToList(strDate);
        }catch (IllegalArgumentException e){
            logger.error("{}",e);
            return 0;
        }

        int year  = 0;
        int month = 0;
        int day   = 0;

        if (splitDate.size() > 0){
            year = StringTools.toInt(splitDate.get(0));
            if (year< 0 || year >9999){
                year =  0;
            }
        }

        if (splitDate.size() > 1){
            month = StringTools.toInt(splitDate.get(1));
            if (month < 0 || month > 12){
                month = 0;
            }
        }

        if (splitDate.size() > 2){
            day = StringTools.toInt(splitDate.get(2));
            if (day < 0 || day > 31){
                day = 0;
            }
        }

        return year*10000 + month * 100 + day;
    }



    /**
     * 获取当前日期，格式:yyyymmdd
     * @return
     */
    public static String currentDateStr(String dateFormat){
        return getDateStr(new Date(),dateFormat);
    }

    public static String currentDateStr2(){
        return currentDateStr("yyyy-MM-dd");
    }



    /**
     * 获取当前日期，格式:yyyymmdd
     * @return
     */
    public static String getDateStr(Date date, String format){
        return getDateStr(date,format,VMS_TIMEZONE);
    }

    public static String getDateStr(Date date, String format,String timeZoneStr){
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        if (!Strings.isNullOrEmpty(timeZoneStr)){
            sdf.setTimeZone(TimeZone.getTimeZone(timeZoneStr));
        }

        return sdf.format(date).toString();
    }

    private static String distanceToday(int type,int distanceMonth){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(type, distanceMonth);
        Date dBefore = calendar.getTime();
        return getDateStr(dBefore,"yyyy-MM-dd");
    }


    public static String distanceTodayByMonth(int distance){
        return distanceToday(Calendar.MONTH, distance);
    }

    public static int distanceTodayByMonthInt(int distance){
        return date2IntVal(distanceTodayByMonth(distance));
    }

    public static String getDateAMonthAgo(){
        return distanceTodayByMonth(-1);
    }

    public static String distanceTodayByDay(int distance){
        return distanceToday(Calendar.DAY_OF_YEAR, distance);
    }


    public static String millis2DateStr(long time){
        Date date = new Date(time);
        return getDateStr(date,"yyyy-MM-dd HH:mm:ss.S");
    }


    public static int currentDateInt(){
        return date2IntVal(currentDateStr2());
    }

    public static int dateAMonthAgoInt(){
        return date2IntVal(getDateAMonthAgo());
    }

    public static Date maxDate(Date d1,Date d2){
        if (d1 == null){
            return d2;
        }

        if (d2 == null){
            return d1;
        }

        if (d1.getTime() > d2.getTime()){
            return d1;
        }else {
            return d2;
        }
    }


    public static boolean outOfTimeLimit(long beforeTs, long limitTS){
        if (beforeTs == 0 || limitTS == 0){
            return false;
        }

        return ((System.currentTimeMillis() - beforeTs) < limitTS);

    }

   public static long getSecondsOfDays(long days){
        return days * SECONDS_OF_DAY;
   }

   public static long dateCeil(long ms){
       long part  = ms % MS_OF_DAY;
       return ms-part+MS_OF_DAY;
   }

    public static long dateCeil(long nowMS,int secondUnit){
        int ms = secondUnit *1000;
        long part  = nowMS % ms;
        return nowMS-part+ms;
    }

    public static long dateFloor(long nowMS, int secondUnit){
        return nowMS - (nowMS % (secondUnit*1000));
    }

    public static long dateFloor(int secondUnit){
        long now = System.currentTimeMillis();
        return dateFloor(now,secondUnit);
    }

    public static int getMaxDayByYearMonth(int year, int month) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, month - 1);
        return calendar.getActualMaximum(Calendar.DATE);
    }

    public static String fmtDate(String dateStr, boolean ceil) {
        if (Strings.isNullOrEmpty(dateStr)) {
            return "";
        }

        List<String> splitDate = Lists.newArrayList();
        try {
            splitDate = Splitter.on("-").trimResults().limit(3).splitToList(dateStr);
        } catch (IllegalArgumentException e) {
            logger.error("{}", e);
            return "";
        }

        int year = 0;
        int month = 0;
        int day = 0;


        if (splitDate.size() > 0) {
            year = StringTools.toInt(splitDate.get(0));
            if (year < 0 || year > 9999) {
                year = 0;
            }
        }

        if (splitDate.size() > 1) {
            month = StringTools.toInt(splitDate.get(1));
            if (month < 0 || month > 12) {
                month = 0;
            }
        }

        if (splitDate.size() > 2) {
            day = StringTools.toInt(splitDate.get(2));
            if (day < 0 || day > 31) {
                day = 0;
            }
        }

        if (ceil){
            year = (year == 0 ? 2037:year);
            month = (month == 0 ? 12:month);
            day = (day == 0 ? getMaxDayByYearMonth(year, month) : day);

        }else {
            year = (year == 0 ? 1970:year);
            month = (month == 0 ? 1:month);
            day = (day==0?1:day);
        }
        return Joiner.on("-").join(year,month,day);
    }

    public static Date getDateFromLocaleStr(String dateStr , String format, Locale locale){
        try {
            DateTimeFormatter fmt = DateTimeFormat.forPattern(format).withLocale(locale);
            DateTime datetime = fmt.parseDateTime(dateStr);
            return datetime.toDate();
        }catch (Exception e){
            logger.error("fmt date [{}] exception:{}",dateStr,e.getMessage());
            return null;
        }
    }
    
}
