package com.valor.mfc.vms.meta.model.database.configuration;

import com.google.common.base.Strings;

import com.valor.mfc.vms.common.database.tool.configruation.AbstractDBCfgWithHikariCP;
import com.valor.mfc.vms.common.tools.type.CollectionUtils;

import common.config.tools.config.ConfigTools2;
import common.config.tools.config.ConfigTools3;

import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import java.util.List;
import java.util.Properties;

import javax.sql.DataSource;

@Configuration
@EnableTransactionManagement
public class MetaDBConfiguration extends AbstractDBCfgWithHikariCP {
    private static final Logger logger = LoggerFactory.getLogger(MetaDBConfiguration.class);

    public static final String DATABASE_MODEL_BASE_PACKAGE = "com.valor.mfc.vms.meta.model";

    private final String aes_key = ConfigTools3.getString("mfc.meta.db.crypt.key");//"Ebw6MyVpHAC23ZMszboDDQ";
    private String host = getDBConfig("mfc.meta.db.host", aes_key);
    private String inst = getDBConfig("mfc.meta.db.inst", aes_key);
    private String user = getDBConfig("mfc.meta.db.user", aes_key);
    private String pass = getDBConfig("mfc.meta.db.pass", aes_key);

    private List<String> appendPackagesToScans = ConfigTools2.getAsList("mfc.meta.db.appendPackagesToScan",",");

    private String getConnectUrl() {
        if (Strings.isNullOrEmpty(host)
            || Strings.isNullOrEmpty(inst)
            || Strings.isNullOrEmpty(user)
            || Strings.isNullOrEmpty(pass)) {
            logger.error("{}-{}-{}-{}", host, inst, user, pass);
            throw new IllegalArgumentException("Invalid parameter");
        }

        return new StringBuilder()
            .append("jdbc:mysql://")
            .append(host)
            .append("/")
            .append(inst)
            .append("?")
            .append("useUnicode=true&characterEncoding=utf8&autoReconnect=true")
            .toString();
    }

    @Primary
    @Bean(destroyMethod = "close", name = "metaDataSource")
    public DataSource metaDataSource() {
        logger.info("Connect database:{}-{}-{}", host, inst, user);
        return getDataSource(getConnectUrl(), user, pass);
    }

    @Bean(destroyMethod = "destroy", name = "metaSessionFactory")
    public FactoryBean<SessionFactory> metaSessionFactory() {
        if (CollectionUtils.isNullOrEmpty(appendPackagesToScans)) {
            return getSessionFactory(metaDataSource(), DATABASE_MODEL_BASE_PACKAGE);
        } else {
            String[] appendPackagesToScanArry = new String[appendPackagesToScans.size()+1];
            for (int i=0;i<appendPackagesToScans.size();i++){
                appendPackagesToScanArry[i] = appendPackagesToScans.get(i);
            }
            appendPackagesToScanArry[appendPackagesToScans.size()] = DATABASE_MODEL_BASE_PACKAGE;
            return getSessionFactory(metaDataSource(), appendPackagesToScanArry);
        }
    }

    @Bean(name = "metaTransactionManager")
    public PlatformTransactionManager metaTransactionManager() throws Exception {
        return getTransactionManager(metaSessionFactory().getObject(), 1800);
    }

    @Override
    public FactoryBean<SessionFactory> getSessionFactory(
        @Qualifier("metaDataSource") DataSource dataSource, String... scanPackages) {
        return super.getSessionFactory(dataSource, scanPackages);
    }

    @Override
    public PlatformTransactionManager getTransactionManager(
        @Qualifier("metaSessionFactory") SessionFactory sessionFactory, int txTimeout)
        throws Exception {
        HibernateTransactionManager transactionManager =
            new HibernateTransactionManager(sessionFactory);
        transactionManager.setDefaultTimeout(txTimeout);
        return transactionManager;
    }

    @Override
    public Properties getDBProperties() {
        Properties props = super.getDBProperties();
        useL2Cache = ConfigTools3.getBoolean("mfc.meta.db.useL2Cache", false);
        logger.info("Use L2 Cache:[{}]", useL2Cache);
        props.put("hibernate.cache.use_second_level_cache", useL2Cache);
        if (useL2Cache) {
            props.put("hibernate.cache.region.factory_class", l2CacheFactory);
            props.put("hibernate.cache.use_query_cache", useQueryCache);
        }

        props.put("hibernate.hbm2ddl.auto", hbm2ddlAuto);
        return props;
    }


}


