package com.valor.mfc.vms.meta.model.database.configuration;

import com.google.common.base.Strings;
import com.valor.mfc.vms.common.database.model.TableShardingParams;
import com.valor.mfc.vms.common.database.tool.configruation.AbstractDBCfgWithSharding;
import com.valor.mfc.vms.common.database.tool.configruation.AuditInterceptor;
import com.zaxxer.hikari.HikariDataSource;
import common.config.tools.config.ConfigTools3;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;

import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import static java.sql.Connection.TRANSACTION_READ_COMMITTED;

/**
 * @author Hunter Chen
 * created on: 2018-08-29
 *
 * Configuration Example. If the dabase is sharded
 */
//@Configuration
public class DBConfigurationExampleForSharding extends AbstractDBCfgWithSharding {

    private final String aes_key = ConfigTools3.getString("mfc.meta.db.crypt.key");//"Ebw6MyVpHAC23ZMszboDDQ";
    private String host = getDBConfig("mfc.meta.db.host", aes_key);
    private String inst = getDBConfig("mfc.meta.db.inst", aes_key);
    private String user = getDBConfig("mfc.meta.db.user", aes_key);
    private String pass = getDBConfig("mfc.meta.db.pass", aes_key);

    private String appendPackagesToScan = ConfigTools3.getString("mfc.meta.db.appendPackagesToScan");
    private final String DATABASE_MODEL_BASE_PACKAGE = "com.valor.mfc.vms.meta.model";
    private String driverClassName = ConfigTools3.getString("db.driverClassName", "com.mysql.jdbc.Driver");
    private String dialect = ConfigTools3.getString("db.dialect", "org.hibernate.dialect.MySQL5Dialect");
    private String validationQuery = ConfigTools3.getString("db.validationQuery", "SELECT 1");
    private String hbm2ddlAuto = ConfigTools3.getString("db.hbm2ddlAuto", "update");
    private boolean showSql = ConfigTools3.getBoolean("db.stat.showSql", false);
    private boolean generateStatistics = ConfigTools3.getBoolean("db.stat.generateStatistics", false);
    private boolean useL2Cache = ConfigTools3.getBoolean("db.l2cache.useL2Cache", false);
    private boolean useQueryCache = ConfigTools3.getBoolean("db.l2cache.pool.useQueryCache", false);
    private String l2CacheFactory = ConfigTools3.getString("db.l2cache.l2CacheFactory", "org.hibernate.cache.ehcache.EhCacheRegionFactory");



    @Override
    protected DataSource createDataSouce() {
        HikariDataSource result = new HikariDataSource();
        result.setDriverClassName(driverClassName);
        result.setJdbcUrl(getConnectUrl());
        result.setUsername(user);
        result.setPassword(pass);
        result.setMaximumPoolSize(30);
        return result;
    }

    @Override
    protected List<TableShardingParams> configureTableShardingParams() {
        List<TableShardingParams>  list = new ArrayList<>();
        TableShardingParams params = new TableShardingParams("linked_user_cloud","testdb.linked_user_cloud_${0..127}","uid","linked_user_cloud_${uid % 128}");
        list.add(params);
        return list;
    }

    @Override
    protected String configureDatabaseName() {
        return "testdb";
    }


    private String getConnectUrl() {
        if (Strings.isNullOrEmpty(host) || Strings.isNullOrEmpty(inst) || Strings.isNullOrEmpty(user) || Strings.isNullOrEmpty(pass)) {
            throw new IllegalArgumentException("Invalid parameter");
        }
        return new StringBuilder().append("jdbc:mysql://").append(host).append("/").append(inst).append("?").append("useUnicode=true&characterEncoding=utf8&autoReconnect=true").toString();
    }


    private Properties getDBProperties() {
        Properties props = getHibernateDBProperties();
        useL2Cache = ConfigTools3.getBoolean("mfc.meta.db.useL2Cache", false);
        props.put("hibernate.cache.use_second_level_cache", useL2Cache);
        if (useL2Cache) {
            props.put("hibernate.cache.region.factory_class", l2CacheFactory);
            props.put("hibernate.cache.use_query_cache", useQueryCache);
        }

        props.put("hibernate.hbm2ddl.auto", hbm2ddlAuto);
        return props;
    }

    @Bean(name = "metaSessionFactory")
    @Primary
    public FactoryBean<SessionFactory> metaSessionFactory() throws SQLException {
        if (Strings.isNullOrEmpty(appendPackagesToScan)) {
            return getSessionFactory(getShardingDataSource(), DATABASE_MODEL_BASE_PACKAGE);
        } else {
            return getSessionFactory(getShardingDataSource(), DATABASE_MODEL_BASE_PACKAGE, appendPackagesToScan);
        }
    }

    @Bean(name = "metaTransactionManager")
    public PlatformTransactionManager metaTransactionManager() throws Exception {
        HibernateTransactionManager transactionManager = new HibernateTransactionManager(metaSessionFactory().getObject());
        transactionManager.setDefaultTimeout(1800);
        return transactionManager;
    }

    private FactoryBean<SessionFactory> getSessionFactory(DataSource dataSource, String... scanPackages) {
        LocalSessionFactoryBean sessionFactoryBean = new LocalSessionFactoryBean();
        sessionFactoryBean.setEntityInterceptor(new AuditInterceptor());
        sessionFactoryBean.setDataSource(dataSource);
        sessionFactoryBean.setPackagesToScan(scanPackages);
        sessionFactoryBean.setHibernateProperties(getDBProperties());
        return sessionFactoryBean;
    }

    private Properties getHibernateDBProperties() {
        Properties props = new Properties();
        props.put("hibernate.dialect", dialect);
        props.put("hibernate.show_sql", showSql);
        props.put("hibernate.format_sql", false);
        props.put("hibernate.generate_statistics", generateStatistics);
        props.put("hibernate.hbm2ddl.auto", hbm2ddlAuto);
        props.put("hibernate.connection.isolation", TRANSACTION_READ_COMMITTED);
        props.put("hibernate.use_sql_comments", true);
        props.put("hibernate.cache.use_second_level_cache", false);
        props.put("hibernate.cache.use_query_cache", false);
        props.put("hibernate.connection.CharSet", "utf8");
        props.put("hibernate.connection.characterEncoding", "utf8");
        props.put("hibernate.connection.useUnicode", true);
        props.put("hibernate.autoReconnect", true);
        return props;
    }

}
