package com.valor.mfc.vms.meta.model.service;

import com.google.common.base.Strings;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;

import com.valor.mfc.vms.api.model.constant.EArtworkType;
import com.valor.mfc.vms.api.model.constant.EMetaSource;
import com.valor.mfc.vms.api.model.meta.media.v1.ArtworkDO;
import com.valor.mfc.vms.common.tools.meta.LanguageTools;
import com.valor.mfc.vms.common.tools.type.CollectionUtils;
import com.valor.mfc.vms.common.tools.type.DdoKeyTools;
import com.valor.mfc.vms.meta.model.database.ddo.media.base.AbstractArtwork;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by Frank.Huang on 2016/11/29.
 */
public class ArtworkQueryService {
    private Map<String,String> masterUrlMap = Maps.newHashMap();
    private Multimap<Long, AbstractArtwork> artworkMultimap = HashMultimap.create();

    public ArtworkQueryService(List<AbstractArtwork> artworks) {

        if (!CollectionUtils.isNullOrEmpty(artworks)){
            for (AbstractArtwork artwork:artworks){
                if (artwork.getMaster() == 1){
                    String key = DdoKeyTools.key(artwork.getId(),artwork.getLanguage(),artwork.getArtworkType());
                    masterUrlMap.put(key,artwork.getUrl());
                }

                artworkMultimap.put(artwork.getId(),artwork);
            }
        }
    }

    public String getMasterArtwork(long id, String language, EArtworkType artworkType){
        String artworkLanguage = LanguageTools.getOrDefaultLanguage(language);
        String key = DdoKeyTools.key(id,artworkLanguage,artworkType);
        String keyDefault = DdoKeyTools.key(id,LanguageTools.LANG_EN,artworkType);
        return masterUrlMap.getOrDefault(key,masterUrlMap.get(keyDefault));
    }

    public String getMasterArtwork(Set<Long> ids, String language, EArtworkType artworkType){
        if (!CollectionUtils.isNullOrEmpty(ids)){
            for (long id : ids){
                String artwork = getMasterArtwork(id,language,artworkType);
                if (!Strings.isNullOrEmpty(artwork)){
                    return artwork;
                }
            }
        }

        return "";
    }

    public String getMasterPoster(long id, String language){
        return getMasterArtwork(id,language,EArtworkType.POSTER);
    }

    public String getMasterBackdrop(long id, String language){
        return getMasterArtwork(id,language,EArtworkType.BACKDROP);
    }

    public String getMasterPoster(Set<Long> ids, String language){
        return getMasterArtwork(ids,language,EArtworkType.POSTER);
    }

    public String getMasterBackdrop(Set<Long> ids, String language){
        return getMasterArtwork(ids,language,EArtworkType.BACKDROP);
    }

    public List<ArtworkDO> getArtworks(long id){
       return getArtworks(id,0,Integer.MAX_VALUE);
    }
    public List<ArtworkDO> getArtworks(long id,int min,int max){
        Collection<AbstractArtwork> artworks = artworkMultimap.get(id);
        if (CollectionUtils.isNullOrEmpty(artworks)){
            return Lists.newArrayList();
        }

        List<AbstractArtwork> artworkList = Lists.newArrayList(artworks);
        artworkList.sort(new Comparator<AbstractArtwork>() {
            @Override
            public int compare(AbstractArtwork o1, AbstractArtwork o2) {
                int score1 = calcArtworkSortScore(o1);
                int score2 = calcArtworkSortScore(o2);

                return score2 - score1;
            }
        });

        int size = artworkList.size();
        if (min >= size){
            return Lists.newArrayList();
        }

        if (max > size){
            max = size -1;
        }

        List<ArtworkDO> artworkDOS = Lists.newArrayList();
        for (AbstractArtwork e:artworkList.subList(min,max)){
           artworkDOS.add(new ArtworkDO(e.getId(),e.getArtworkType(),e.getUrl(), e.getMaster()));
        }

        return artworkDOS;
    }

    private int calcArtworkSortScore(AbstractArtwork artwork){
        final int SORT_FACTOR_MASTER = 3000;
        final int SORT_FACTOR_TYPE = 2000;
        final int SORT_BACKDROP_SOURCE_MFC = 2000;
        final int SORT_BACKDROP_SOURCE_TMDB = 1000;

        if (artwork==null){
            return 0;
        }

        int score = 0;
        if (artwork.getMaster() == 1){
            score += SORT_FACTOR_MASTER;
        }

        if (artwork.getArtworkType()==EArtworkType.POSTER){
            score += SORT_FACTOR_TYPE;
        }

        if (artwork.getArtworkType()==EArtworkType.BACKDROP){
            if (artwork.getSource()== EMetaSource.TMDB){
                score += SORT_BACKDROP_SOURCE_TMDB;
            }

            if (artwork.getSource()== EMetaSource.VXMFC){
                score += SORT_BACKDROP_SOURCE_MFC;
            }
        }

        return score;

    }
}
