/*
 *      Copyright (c) 2004-2015 YAMJ Members
 *      https://github.com/organizations/YAMJ/teams
 *
 *      This file is part of the Yet Another Media Jukebox (YAMJ).
 *
 *      YAMJ is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, either version 3 of the License, or
 *      any later version.
 *
 *      YAMJ is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with YAMJ.  If not, see <http://www.gnu.org/licenses/>.
 *
 *      Web: https://github.com/YAMJ/yamj-v3
 *
 */
package com.valor.mfc.vms.meta.model.database.dao;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import common.base.tools.stat.StatTools;
import com.valor.mfc.vms.common.database.tool.access.SqlScalars;
import com.valor.mfc.vms.common.database.tool.access.constants.EOperator;
import com.valor.mfc.vms.common.database.tool.access.constants.ESqlOrder;
import com.valor.mfc.vms.common.tools.type.CollectionUtils;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * Hibernate DAO implementation
 */
@Repository
@Transactional(value = "metaTransactionManager")
public class AbstractMetaBaseDao {
    private static final Logger logger = LoggerFactory.getLogger(AbstractMetaBaseDao.class);

    @Autowired
    @Qualifier(value = "metaSessionFactory")
    protected SessionFactory metaSessionFactory;

    //protected SessionFactory sessionFactory;

    public void setSessionFactory(SessionFactory sessionFactory) {
        this.metaSessionFactory = sessionFactory;
    }

    @Transactional(value = "metaTransactionManager")
    public Session currentSession() {
        return metaSessionFactory.getCurrentSession();
//        try {
//            return sessionFactory.getCurrentSession();
//        }catch (HibernateException e){
//            return sessionFactory.openSession();
//        }
    }

    /**
     * Flush and clear the session.
     */
    public void flushAndClear() {
        Session session = currentSession();
        session.flush();
        session.clear();
    }

    /**
     * store an entity.
     *
     * @param entity the entity to store
     */
    public void storeEntity(final Object entity) {
        if (entity != null) {
            currentSession().saveOrUpdate(entity);
        }
    }

    /**
     * store an entity if absent.
     *
     * @param entity the entity to store
     */
    public void storeEntityIfAbsent(final Object entity) {
        Session session = currentSession();
        doStoreEntityIfAbsent(entity, session);
    }

    private void doStoreEntityIfAbsent(final Object entity, Session session) {
        boolean exists = session.contains(entity);
        if (!exists) {
            session.saveOrUpdate(entity);
        }
    }

    /**
     * Merge an entity.
     *
     * @param entity the entity to merge
     */
    public Object mergeEntity(final Object entity) {
        return currentSession().merge(entity);
    }

    public void mergeAll(final Collection entities) {
        if (entities != null && !entities.isEmpty()) {
            Session session = currentSession();
            for (Object entity : entities) {
                session.merge(entity);
            }
        }
    }

    /**
     * store all entities.
     *
     * @param entities the entities to store
     */
    @SuppressWarnings("rawtypes")
    public void storeAll(final Collection entities) {
        if (entities != null && !entities.isEmpty()) {
            Session session = currentSession();
            for (Object entity : entities) {
                session.saveOrUpdate(entity);
            }
        }
    }

    /**
     * store every entity if absent.
     *
     * @param entities the entities to store
     */
    @SuppressWarnings("rawtypes")
    public void storeAllIfAbsent(final Collection entities) {
        if (entities != null && !entities.isEmpty()) {
            Session session = currentSession();
            for (Object entity : entities) {
                doStoreEntityIfAbsent(entity, session);
            }
        }
    }

    /**
     * Delete all entities.
     *
     * @param entities the entities to delete
     */
    @SuppressWarnings("rawtypes")
    public void deleteAll(final Collection entities) {
        if (entities != null && !entities.isEmpty()) {
            Session session = currentSession();
            for (Object entity : entities) {
                session.delete(entity);
            }
        }
    }

    /**
     * Save an entity.
     *
     * @param entity the entity to save
     */
    public void saveEntity(final Object entity) {
        currentSession().save(entity);
    }

    /**
     * Update an entity.
     *
     * @param entity the entity to update
     */
    public void updateEntity(final Object entity) {
        currentSession().update(entity);
    }

    /**
     * Delete an entity.
     *
     * @param entity the entity to delete
     */
    public void deleteEntity(final Object entity) {
        currentSession().delete(entity);
    }

    /**
     * Get a single object by its id
     */
    @SuppressWarnings("unchecked")
    public <T> T getById(Class<T> entityClass, Serializable id) {
        return (T) currentSession().get(entityClass, id);
    }

    /**
     * Get a single object by the passed field using the name case sensitive.
     */
    @SuppressWarnings("unchecked")
    public <T> T getByNaturalId(Class<? extends T> entityClass, String field, String name) {
        return (T) currentSession().byNaturalId(entityClass).using(field, name).load();
    }

    /**
     * Get a single object by the passed field using the name case insensitive.
     */
    @SuppressWarnings("unchecked")
    public <T> T getByNaturalIdCaseInsensitive(Class<? extends T> entityClass, String field, String name) {
        StringBuilder sb = new StringBuilder();
        sb.append("from ");
        sb.append(entityClass.getSimpleName());
        sb.append(" where lower(").append(field).append(") = :name) { ");

        Map<String, Object> params = Collections.singletonMap("name", (Object) name.toLowerCase());
        return (T) this.findUniqueByNamedParameters(sb, params);
    }

    /**
     * Convert row object to a string.
     *
     * @return <code>String</code>
     */
    protected static String convertRowElementToString(Object rowElement) {
        if (rowElement == null) {
            return null;
        } else if (rowElement instanceof String) {
            return (String) rowElement;
        } else {
            return rowElement.toString();
        }
    }

    /**
     * Convert row object to Integer.
     *
     * @return <code>Integer</code>
     */
    protected static Integer convertRowElementToInteger(Object rowElement) {
        if (rowElement == null) {
            return 0;
        } else if (StringUtils.isNumeric(rowElement.toString())) {
            return Integer.valueOf(rowElement.toString());
        } else {
            return 0;
        }
    }

    /**
     * Convert row object to Long.
     *
     * @return <code>Long</code>
     */
    protected static Long convertRowElementToLong(Object rowElement) {
        if (rowElement == null) {
            return 0L;
        } else if (StringUtils.isNumeric(rowElement.toString())) {
            return Long.valueOf(rowElement.toString());
        } else {
            return 0L;
        }
    }

    /**
     * Convert row object to date.
     */
    protected static Date convertRowElementToDate(Object rowElement) {
        if (rowElement == null) {
            return null;
        } else if (rowElement instanceof Date) {
            return (Date) rowElement;
        } else {
            // TODO invalid date
            return null;
        }
    }

    /**
     * Convert row object to date.
     */
    protected static Timestamp convertRowElementToTimestamp(Object rowElement) {
        if (rowElement == null) {
            return null;
        } else if (rowElement instanceof Timestamp) {
            return (Timestamp) rowElement;
        } else {
            // TODO invalid timestamp
            return null;
        }
    }

    /**
     * Convert row object to big decimal.
     *
     * @return <code>BigDecimal</code>
     */
    protected static BigDecimal convertRowElementToBigDecimal(Object rowElement) {
        if (rowElement == null) {
            return BigDecimal.ZERO;
        } else if (rowElement instanceof BigDecimal) {
            return (BigDecimal) rowElement;
        } else {
            return new BigDecimal(rowElement.toString());
        }
    }

    @SuppressWarnings("rawtypes")
    protected static void applyNamedParameterToQuery(Query queryObject, String paramName, Object value) throws HibernateException {
        if (value instanceof Collection) {
            queryObject.setParameterList(paramName, (Collection) value);
        } else if (value instanceof Object[]) {
            queryObject.setParameterList(paramName, (Object[]) value);
        } else if (value instanceof String) {
            queryObject.setString(paramName, (String) value);
        } else {
            queryObject.setParameter(paramName, value);
        }
    }

    /**
     * Find entries-
     *
     * @param queryString the query string
     * @return list of entities
     */
    @SuppressWarnings("rawtypes")
    public List find(CharSequence queryString) {
        Query queryObject = currentSession().createQuery(queryString.toString());
        queryObject.setCacheable(true);
        return queryObject.list();
    }

    /**
     * Find entries by id.
     *
     * @param queryString the query string
     * @param id          the id
     * @return list of entities
     */
    @SuppressWarnings("rawtypes")
    public List findById(CharSequence queryString, Long id) {
        Query queryObject = currentSession().createQuery(queryString.toString());
        queryObject.setCacheable(true);
        queryObject.setParameter("id", id);
        return queryObject.list();
    }

    /**
     * Find list of entities by named parameters.
     *
     * @param queryCharSequence the query string
     * @param params            the named parameters
     * @return list of entities
     */
    @SuppressWarnings("rawtypes")
    public List findByNamedParameters(CharSequence queryCharSequence, Map<String, Object> params) {
        Query query = currentSession().createQuery(queryCharSequence.toString());
        query.setCacheable(true);
        for (Entry<String, Object> param : params.entrySet()) {
            applyNamedParameterToQuery(query, param.getKey(), param.getValue());
        }
        return query.list();
    }

    /**
     * Find unique entity by named parameters.
     *
     * @param queryCharSequence the query string
     * @param params            the named parameters
     * @return list of entities
     */
    public Object findUniqueByNamedParameters(CharSequence queryCharSequence, Map<String, Object> params) {
        Query query = currentSession().createQuery(queryCharSequence.toString());
        query.setCacheable(true);
        for (Entry<String, Object> param : params.entrySet()) {
            applyNamedParameterToQuery(query, param.getKey(), param.getValue());
        }
        return query.uniqueResult();
    }

    /**
     * Execute an update statement.
     *
     * @param queryCharSequence the query string
     * @return number of affected rows
     */
    public int executeUpdate(CharSequence queryCharSequence) {
        Query query = currentSession().createQuery(queryCharSequence.toString());
        query.setCacheable(true);
        return query.executeUpdate();
    }

    /**
     * Execute an update statement.
     *
     * @param queryCharSequence the query string
     * @param params            the named parameters
     * @return number of affected rows
     */
    public int executeUpdate(CharSequence queryCharSequence, Map<String, Object> params) {
        Query query = currentSession().createQuery(queryCharSequence.toString());
        query.setCacheable(true);
        for (Entry<String, Object> param : params.entrySet()) {
            applyNamedParameterToQuery(query, param.getKey(), param.getValue());
        }
        return query.executeUpdate();
    }

    /**
     * Execute a SQL update statement.
     *
     * @param queryCharSequence the query string
     * @return number of affected rows
     */
    public int executeSqlUpdate(CharSequence queryCharSequence) {
        SQLQuery query = currentSession().createSQLQuery(queryCharSequence.toString());
        query.setCacheable(true);
        return query.executeUpdate();
    }

    /**
     * Execute an update statement.
     *
     * @param queryCharSequence the query string
     * @param params            the named parameters
     * @return number of affected rows
     */
    public int executeSqlUpdate(CharSequence queryCharSequence, Map<String, Object> params) {
        SQLQuery query = currentSession().createSQLQuery(queryCharSequence.toString());
        query.setCacheable(true);
        for (Entry<String, Object> param : params.entrySet()) {
            applyNamedParameterToQuery(query, param.getKey(), param.getValue());
        }
        return query.executeUpdate();
    }

    private void setResultTransformer(Class T, SQLQuery query) {
        if (T != null) {
            if (T.equals(String.class)) {
                // no transformer needed
            } else if (T.equals(Long.class)) {
                // no transformer needed
            } else if (T.equals(Integer.class)) {
                // no transformer needed
            } else if (T.equals(Object[].class)) {
                // no transformer needed
            } else {
                query.setResultTransformer(Transformers.aliasToBean(T));
            }
        }
    }

    public <T> List<T> query(Class T, SqlScalars sqlScalars, int firstResult, int maxResult) {
        SQLQuery query = sqlScalars.createSqlQuery(currentSession());

        query.setReadOnly(true);
//        query.setCacheable(false);
        setResultTransformer(T, query);

        sqlScalars.populateScalars(query);
        query.setFirstResult(firstResult);
        query.setMaxResults(maxResult);

        return query.list();
    }


    public <T> List<T> query(Class T, SqlScalars sqlScalars, boolean cacheable) {
        SQLQuery query = sqlScalars.createSqlQuery(currentSession());
        query.setReadOnly(true);
        query.setCacheable(cacheable);
        setResultTransformer(T, query);
        sqlScalars.populateScalars(query);
//        query.setFirstResult(firstResult);
//        query.setMaxResults(maxResult);

        Object result = query.list();
        return (List<T>) result;
    }

    public List getDataList(boolean enableCache,
                            Class returnType,
                            Map<EOperator, Map<String, Object>> filters) {

        return Lists.newArrayList();
    }

    /**
     * 获取简单的list列表
     */
    public List getListData(boolean enableCache, Class returnType, Map<String, Object> filters) {
        Criteria criteria = currentSession().createCriteria(returnType);
        criteria.setCacheable(enableCache);
        if (filters != null) {
            for (Entry entry : filters.entrySet()) {
                criteria.add(Restrictions.eq((String) entry.getKey(), entry.getValue()));
            }
        }

        List list = criteria.list();
        return list;
    }

    public List getListDataIn(boolean enableCache, Class returnType, Map<String, Object> filters) {
        Criteria criteria = currentSession().createCriteria(returnType);
        criteria.setCacheable(enableCache);
        if (filters != null) {
            for (Entry entry : filters.entrySet()) {
                if (entry.getValue() instanceof Collection) {
                    criteria.add(Restrictions.in((String) entry.getKey(), ((Collection) entry.getValue()).toArray()));
                } else {
                    criteria.add(Restrictions.eq((String) entry.getKey(), entry.getValue()));
                }
            }
        }

        List list = criteria.list();
        return list;
    }

    /**
     * 获取Set列表
     */
    public Set getSetData(boolean enableCache, Class returnType, Map<String, Object> filters) {
        return new HashSet<>(getListData(enableCache, returnType, filters));
    }

    /**
     * 获取条件的过滤值
     */
    public Object getData(boolean enableCahce, Class returnType, Map<String, Object> filters) {
        Criteria criteria = currentSession().createCriteria(returnType);
        criteria.setCacheable(enableCahce);
        if (filters != null) {
            for (Entry entry : filters.entrySet()) {
                criteria.add(Restrictions.eq((String) entry.getKey(), entry.getValue()));
            }
        }

        return criteria.uniqueResult();
    }

    public int excuteSql(String sql) {
        return currentSession().createSQLQuery(sql).executeUpdate();
    }

    public List<Object[]> excuteSql2(String sql) {
        return currentSession().createSQLQuery(sql).list();
    }

    public int excuteQueryCount(String sql) {
        try {
            return ((BigInteger) currentSession().createSQLQuery(sql).uniqueResult()).intValue();
        } catch (Exception e) {
            logger.info("query count exception! sql={} exception:{}", sql, e);
            return -1;
        }
    }

    public long criteriaQueryCount(Class valueType) {
        return criteriaQueryCount(valueType, false);
    }

    public long criteriaQueryCount(Class valueType, boolean cacheable) {
        StatTools.startDBTimer("count-" + valueType.getSimpleName());
        Long ret = (Long) currentSession().createCriteria(valueType)
                .setReadOnly(true).setCacheable(cacheable)
                .setProjection(Projections.rowCount())
                .uniqueResult();
        StatTools.startDBTimer("count-" + valueType.getSimpleName());
        return ret;
    }

    public long criteriaQueryCount(Class valueType, Map<String, Object> filter) {
        Criteria criteria = currentSession().createCriteria(valueType);
        if (filter != null) {
            for (Entry entry : filter.entrySet()) {
                if (entry.getValue() instanceof Collection) {
                    criteria.add(Restrictions.in((String) entry.getKey(), (Collection) entry.getValue()));
                } else {
                    criteria.add(Restrictions.eq((String) entry.getKey(), entry.getValue()));
                }
            }
        }
        criteria.setProjection(Projections.rowCount());
        Long ret = (Long) criteria.uniqueResult();
        return ret;
    }

    public List getPageData(Class returnType,
                            Map<String, Object> filters,
                            Map<String, ESqlOrder> order,
                            int firstResult,
                            int maxResult) {
        return getPageData(false, returnType, filters, order, firstResult, maxResult);
    }


    public List getPageData(boolean cacheable,
                            Class returnType,
                            Map<String, Object> filters,
                            Map<String, ESqlOrder> order,
                            int firstResult,
                            int maxResult) {
        Criteria criteria = currentSession().createCriteria(returnType);
        criteria.setFirstResult(firstResult);
        criteria.setMaxResults(maxResult);
        criteria.setReadOnly(true);
        criteria.setCacheable(cacheable);

        if (filters != null) {
            for (Entry entry : filters.entrySet()) {
                criteria.add(Restrictions.eq((String) entry.getKey(), entry.getValue()));
            }
        }

        if (!CollectionUtils.isNullOrEmpty(order)) {
            order.forEach((k, v) -> {
                if (v == ESqlOrder.ASC) {
                    criteria.addOrder(Order.asc(k));
                } else if (v == ESqlOrder.DESC) {
                    criteria.addOrder(Order.desc(k));
                }
            });
        }

        List list = criteria.list();
        return list;
    }

    public String getLatestUpdateTime(Class clazz, Map<String, Object> filters) {
        return String.valueOf(getLatestUpdateTS(clazz, filters));
    }

    public Date getLatestUpdateDate(Class clazz, Map<String, Object> filters) {
        Criteria criteria = currentSession().createCriteria(clazz);
        if (filters != null) {
            for (Entry entry : filters.entrySet()) {
                if (entry.getValue() instanceof Collection) {
                    criteria.add(Restrictions.in((String) entry.getKey(), (Collection) entry.getValue()));
                } else {
                    criteria.add(Restrictions.eq((String) entry.getKey(), entry.getValue()));
                }
            }
        }

        criteria.setProjection(Projections.projectionList().add(Projections.max("lastModifyTime")));
        return (Date) criteria.uniqueResult();
    }

    public long getLatestUpdateTS(Class clazz, Map<String, Object> filters) {
        Date date = getLatestUpdateDate(clazz, filters);
        if (date == null) {
            return 0L;
        } else {
            return date.getTime();
        }
    }

    public String queryDBStamp(Class clazz, String lastUpdateField) {
        if (Strings.isNullOrEmpty(lastUpdateField)) {
            lastUpdateField = "lastModifyTime";
        }

        Criteria criteria = currentSession().createCriteria(clazz);
        criteria.setCacheable(false);
        criteria.setReadOnly(true);
        criteria.setProjection(Projections.projectionList().add(Projections.count(lastUpdateField))
                .add(Projections.max(lastUpdateField)));

        Object[] objList = (Object[]) criteria.uniqueResult();
        if (objList == null){
            return "";
        } else {
            return Joiner.on("@").useForNull("-").join(objList);
        }


    }

}

