package com.valor.vod.hotkey.client;

import com.valor.vod.hotkey.client.callback.ReceiveNewKeySubscribe;
import com.valor.vod.hotkey.client.configcenter.ConfigCenterFactory;
import com.valor.vod.hotkey.client.configcenter.ConfigCenterStarter;
import com.valor.vod.hotkey.client.core.eventbus.EventBusCenter;
import com.valor.vod.hotkey.client.core.key.PushSchedulerStarter;
import com.valor.vod.hotkey.client.core.rule.KeyRuleHolder;
import com.valor.vod.hotkey.client.core.worker.WorkerChangeSubscriber;
import com.valor.vod.hotkey.client.core.worker.WorkerRetryConnector;
import com.valor.vod.hotkey.common.configcenter.ConfigCenterCfg;
import com.valor.vod.hotkey.common.tool.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.*;

/**
 * 客户端启动器
 *
 * @author wuweifeng wrote on 2019-12-05
 * @version 1.0
 */
public class ClientStarter {

    private static final Logger logger = LoggerFactory.getLogger(ClientStarter.class);

    private String etcdServer;
    /** 推送key的间隔(毫秒)，推送越快，探测的越密集，会越快探测出来，但对client资源消耗相应增大 */
    private Long pushPeriod;
    /** caffeine的最大容量，默认给5万 */
    private int caffeineSize;
    /** 配置中心 */
    private ConfigCenterCfg configCenterCfg;

    public ClientStarter(String appName) {
        if (appName == null) {
            throw new NullPointerException("APP_NAME cannot be null!");
        }
        Context.APP_NAME = appName;
    }

    public void setConfigCenterCfg(ConfigCenterCfg configCenterCfg) {
        this.configCenterCfg = configCenterCfg;
    }

    /** 启动监听etcd */
    public void startPipeline() {
        // 设置caffeine的最大容量
        Context.CAFFEINE_SIZE = caffeineSize;

        // 新建线程池
        ScheduledExecutorService scheduledExecutor =
                new ScheduledThreadPoolExecutor(
                        4, new NamedThreadFactory("hotkey-client-executor-", true));

        // 开始定时推送
        PushSchedulerStarter pushSchedulerStarter = new PushSchedulerStarter(scheduledExecutor);
        pushSchedulerStarter.startPusher(pushPeriod);
        pushSchedulerStarter.startCountPusher(10);

        // 开启worker重连器
        WorkerRetryConnector workerRetryConnector = new WorkerRetryConnector(scheduledExecutor);
        workerRetryConnector.retryConnectWorkers();

        // 注册事件总线
        registerEventBus();

        // 连接注册中心
        retryConnectConfigCenter(0, 0, scheduledExecutor);
    }

    /**
     * 尝试重新
     *
     * @param delay 延迟重新
     * @param retry 重试次数
     * @param scheduledExecutor executor
     */
    private void retryConnectConfigCenter(
            long delay, int retry, ScheduledExecutorService scheduledExecutor) {
        scheduledExecutor.schedule(
                () -> {
                    try {
                        if (configCenterCfg != null) {
                            ConfigCenterFactory.buildConfigCenter(configCenterCfg);
                        } else {
                            ConfigCenterFactory.buildConfigCenter(etcdServer);
                        }
                    } catch (Throwable e) {
                        logger.error("ConfigCenter unconnected. retry:{}", retry, e);
                        // 过10s后再重新连接
                        retryConnectConfigCenter(10000, retry + 1, scheduledExecutor);
                        return;
                    }

                    // ConfigCenter相关监听开启
                    ConfigCenterStarter configCenterStarter =
                            new ConfigCenterStarter(scheduledExecutor);
                    configCenterStarter.start();
                },
                delay,
                TimeUnit.MILLISECONDS);
    }

    private void registerEventBus() {
        // netty连接器会关注WorkerInfoChangeEvent事件
        EventBusCenter.register(new WorkerChangeSubscriber());
        // 热key探测回调关注热key事件
        EventBusCenter.register(new ReceiveNewKeySubscribe());
        // Rule的变化的事件
        EventBusCenter.register(new KeyRuleHolder());
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String appName;
        private String etcdServer;
        private Long pushPeriod;
        private int caffeineSize = 200000;
        private ConfigCenterCfg configCenterCfg;

        public Builder() {}

        public Builder setAppName(String appName) {
            this.appName = appName;
            return this;
        }

        public Builder setCaffeineSize(int caffeineSize) {
            if (caffeineSize < 128) {
                caffeineSize = 128;
            }
            this.caffeineSize = caffeineSize;
            return this;
        }

        public Builder setEtcdServer(String etcdServer) {
            this.etcdServer = etcdServer;
            return this;
        }

        public Builder setPushPeriod(Long pushPeriod) {
            this.pushPeriod = pushPeriod;
            return this;
        }

        public Builder setConfigCenterCfg(ConfigCenterCfg configCenterCfg) {
            this.configCenterCfg = configCenterCfg;
            return this;
        }

        public ClientStarter build() {
            ClientStarter clientStarter = new ClientStarter(appName);
            clientStarter.pushPeriod = pushPeriod;
            clientStarter.caffeineSize = caffeineSize;
            clientStarter.etcdServer = etcdServer;
            clientStarter.configCenterCfg = configCenterCfg;
            return clientStarter;
        }
    }
}
