package com.valor.vod.hotkey.client.configcenter;

import com.valor.vod.hotkey.client.Context;
import com.valor.vod.hotkey.client.callback.ReceiveNewKeyEvent;
import com.valor.vod.hotkey.client.core.eventbus.EventBusCenter;
import com.valor.vod.hotkey.client.core.rule.KeyRuleInfoChangeEvent;
import com.valor.vod.hotkey.client.core.worker.WorkerInfoChangeEvent;
import com.valor.vod.hotkey.client.core.worker.WorkerInfoHolder;
import com.valor.vod.hotkey.client.log.JdLogger;
import com.valor.vod.hotkey.common.configcenter.*;
import com.valor.vod.hotkey.common.configcenter.model.*;
import com.valor.vod.hotkey.common.model.HotKeyModel;
import com.valor.vod.hotkey.common.rule.KeyRule;
import com.valor.vod.hotkey.common.tool.*;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * etcd连接管理器
 *
 * @author wuweifeng wrote on 2019-12-10
 * @version 1.0
 */
public class ConfigCenterStarter {

    private final ScheduledExecutorService scheduledExecutor;

    public ConfigCenterStarter(ScheduledExecutorService scheduledExecutor) {
        this.scheduledExecutor = scheduledExecutor;
    }

    public void start() {
        fetchWorkerInfo();

        fetchRule(0);

        //        startWatchWorker();

        startWatchRule();

        // 监听热key事件，只监听手工添加、删除的key
        startWatchHotKey();
    }

    /** 启动后先拉取已存在的热key（来自于手工添加的目录） */
    private void fetchExistHotKey() {
        JdLogger.info(getClass(), "--- begin fetch exist hotKey from etcd ----");
        IConfigCenter configCenter = ConfigCenterFactory.configCenter();
        try {
            // 获取所有热key
            List<KvData> kvDataList =
                    configCenter.getKvDataPrefix(ConfigConstant.hotKeyPath + Context.APP_NAME);
            for (KvData kvData : kvDataList) {
                String key =
                        kvData.getKey()
                                .replace(ConfigConstant.hotKeyPath + Context.APP_NAME + "/", "");
                HotKeyModel model = new HotKeyModel();
                model.setRemove(false);
                model.setKey(key);
                EventBusCenter.getInstance().post(new ReceiveNewKeyEvent(model));
            }
        } catch (RuntimeException ex) {
            // etcd连不上
            JdLogger.error(getClass(), "etcd connected fail. Check the etcd address!!!");
        }
    }

    /** 每隔30秒拉取worker信息 */
    private void fetchWorkerInfo() {
        // 开启拉取etcd的worker信息，如果拉取失败，则定时继续拉取
        scheduledExecutor.scheduleAtFixedRate(
                () -> {
                    JdLogger.info(getClass(), "trying to connect to etcd and fetch worker info");
                    fetch();
                },
                0,
                30,
                TimeUnit.SECONDS);
    }

    private void fetch() {
        IConfigCenter configCenter = ConfigCenterFactory.configCenter();
        try {
            // 获取所有worker的ip
            List<KvData> kvs =
                    configCenter.getKvDataPrefix(ConfigConstant.workersPath + Context.APP_NAME);
            // worker为空，可能该APP没有自己的worker集群，就去连默认的，如果默认的也没有，就不管了，等着心跳
            if (CollectionUtil.isEmpty(kvs)) {
                kvs = configCenter.getKvDataPrefix(ConfigConstant.workersPath + Constant.DEFAULT);
            }
            // 全是空，给个警告
            if (CollectionUtil.isEmpty(kvs)) {
                JdLogger.warn(getClass(), "very important warn !!! workers ip info is null!!!");
            }

            List<String> addresses = new ArrayList<>();
            if (CollectionUtil.isNotEmpty(kvs)) {
                for (KvData kv : kvs) {
                    // value里放的是ip地址
                    String ipPort = kv.getValue();
                    addresses.add(ipPort);
                }
            }

            JdLogger.info(
                    getClass(),
                    "worker info list is : "
                            + addresses
                            + ", now addresses is "
                            + WorkerInfoHolder.getWorkers());
            // 发布worker info变更信息
            notifyWorkerChange(addresses);
        } catch (RuntimeException ex) {
            // etcd连不上
            JdLogger.error(getClass(), "etcd connected fail. Check the etcd address!!!");
        }
    }

    private void notifyWorkerChange(List<String> addresses) {
        EventBusCenter.getInstance().post(new WorkerInfoChangeEvent(addresses));
    }

    private void notifyRuleChange(List<KeyRule> rules) {
        EventBusCenter.getInstance().post(new KeyRuleInfoChangeEvent(rules));
    }

    /** 异步开始监听热key变化信息，该目录里只有手工添加的key信息 */
    private void startWatchHotKey() {
        IConfigCenter configCenter = ConfigCenterFactory.configCenter();
        configCenter.subscribe(
                ConfigConstant.hotKeyPath + Context.APP_NAME,
                new INotifyListener() {
                    @Override
                    @SuppressWarnings("all")
                    public void notifyDataChange(DataEvent event) throws Exception {
                        scheduledExecutor.submit(
                                () -> {
                                    JdLogger.info(getClass(), "--- begin watch hotKey change ----");
                                    KvData kv = event.getKvData();
                                    String key =
                                            kv.getKey()
                                                    .replace(
                                                            ConfigConstant.hotKeyPath
                                                                    + Context.APP_NAME
                                                                    + "/",
                                                            "");

                                    // 如果是删除key，就立刻删除
                                    if (EventType.DELETE == event.getType()) {
                                        HotKeyModel model = new HotKeyModel();
                                        model.setRemove(true);
                                        model.setKey(key);
                                        EventBusCenter.getInstance()
                                                .post(new ReceiveNewKeyEvent(model));
                                    } else {
                                        HotKeyModel model = new HotKeyModel();
                                        model.setRemove(false);
                                        String value = kv.getValue();
                                        // 新增热key
                                        JdLogger.info(
                                                getClass(),
                                                "etcd receive new key : "
                                                        + key
                                                        + " --value:"
                                                        + value);
                                        // 如果这是一个删除指令，就什么也不干
                                        if (Constant.DEFAULT_DELETE_VALUE.equals(value)) {
                                            return;
                                        }

                                        // 手工创建的value是时间戳
                                        model.setCreateTime(Long.parseLong(kv.getValue()));

                                        model.setKey(key);
                                        EventBusCenter.getInstance()
                                                .post(new ReceiveNewKeyEvent(model));
                                    }
                                });
                    }
                });
    }

    private void fetchRule(long delaySecond) {
        // 开启拉取etcd的worker信息，如果拉取失败，则定时继续拉取
        scheduledExecutor.schedule(
                () -> {
                    JdLogger.info(getClass(), "trying to fetch rule info");
                    boolean success = fetchRuleFromEtcd();
                    if (success) {
                        // 拉取已存在的热key
                        fetchExistHotKey();
                    } else {
                        // 5s后再拉取
                        fetchRule(5);
                    }
                },
                delaySecond,
                TimeUnit.SECONDS);
    }

    private boolean fetchRuleFromEtcd() {
        IConfigCenter configCenter = ConfigCenterFactory.configCenter();
        try {
            List<KeyRule> ruleList = new ArrayList<>();
            // 从etcd获取自己的rule
            String rules = configCenter.get(ConfigConstant.rulePath + Context.APP_NAME);
            if (StrUtil.isEmpty(rules)) {
                JdLogger.warn(getClass(), "rule is empty");
                // 会清空本地缓存队列
                notifyRuleChange(ruleList);
                return true;
            }
            ruleList = FastJsonUtils.toList(rules, KeyRule.class);

            notifyRuleChange(ruleList);
            return true;
        } catch (RuntimeException ex) {
            // etcd连不上
            JdLogger.error(getClass(), "etcd connected fail. Check the etcd address!!!");
            return false;
        } catch (Exception e) {
            JdLogger.error(getClass(), "fetch rule failure, please check the rule info in etcd");
            return true;
        }
    }

    /** 异步监听rule规则变化 */
    private void startWatchRule() {
        IConfigCenter configCenter = ConfigCenterFactory.configCenter();
        configCenter.subscribe(
                ConfigConstant.rulePath + Context.APP_NAME,
                new INotifyListener() {
                    @Override
                    @SuppressWarnings("all")
                    public void notifyDataChange(DataEvent event) throws Exception {
                        scheduledExecutor.submit(
                                () -> {
                                    JdLogger.info(getClass(), "--- begin watch rule change ----");
                                    fetchRuleFromEtcd();
                                });
                    }
                });
    }
}
