package com.valor.vod.hotkey.common.configcenter;

import com.valor.vod.hotkey.common.configcenter.model.DataEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.*;

/**
 * @author Bruce Wu
 * @since 2022-12-13
 */
public abstract class AbstractConfigCenter implements IConfigCenter {

    private static final Logger logger = LoggerFactory.getLogger(AbstractConfigCenter.class);

    /** URL listener map */
    private final ConcurrentMap<SubscribeKey, Set<INotifyListener>> listenerMap =
            new ConcurrentHashMap<>();

    @Override
    public void subscribe(String key, INotifyListener listener) {
        SubscribeKey sk = new SubscribeKey(key, false);
        addNotifyListener(sk, listener);
    }

    @Override
    public void subscribePrefix(String key, INotifyListener listener) {
        SubscribeKey sk = new SubscribeKey(key, true);
        addNotifyListener(sk, listener);
    }

    @Override
    public void unsubscribe(String key, INotifyListener listener) {
        SubscribeKey sk = new SubscribeKey(key, false);
        removeNotifyListener(sk, listener);
    }

    private void addNotifyListener(SubscribeKey sk, INotifyListener listener) {
        listenerMap.compute(
                sk,
                (k, listeners) -> {
                    if (listeners == null) {
                        listeners = new CopyOnWriteArraySet<>();
                        listeners.add(listener);
                        doSubscribe(sk);
                    } else {
                        listeners.add(listener);
                    }
                    return listeners;
                });
    }

    private void removeNotifyListener(SubscribeKey sk, INotifyListener listener) {
        listenerMap.computeIfPresent(
                sk,
                (k, listeners) -> {
                    if (listeners.remove(listener) && listeners.isEmpty()) {
                        doUnsubscribe(sk);
                    }
                    return listeners;
                });
    }

    /**
     * @param sk
     */
    protected void doSubscribe(SubscribeKey sk) {}

    protected void doUnsubscribe(SubscribeKey sk) {}

    /**
     * 节点数据改变
     *
     * @param event
     */
    protected void fireDataChange(final DataEvent event) {
        if (logger.isDebugEnabled()) {
            logger.debug("Fire data change" + event);
        }
        listenerMap.forEach(
                (k, v) -> {
                    String path = event.getSource().toString();
                    boolean notify =
                            Objects.equals(path, k.getKey())
                                    || (k.isPrefix() && path.startsWith(k.getKey()));
                    if (notify) {
                        for (INotifyListener listener : v) {
                            try {
                                listener.notifyDataChange(event);
                            } catch (Exception e) {
                                logger.error("Notify error.", e);
                            }
                        }
                    }
                });
    }

    public static class SubscribeKey implements Serializable {
        private String key;
        private boolean prefix;

        public SubscribeKey(String key, boolean prefix) {
            this.key = key;
            this.prefix = prefix;
        }

        public String getKey() {
            return key;
        }

        public boolean isPrefix() {
            return prefix;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SubscribeKey that = (SubscribeKey) o;
            return prefix == that.prefix && Objects.equals(key, that.key);
        }

        @Override
        public int hashCode() {
            return Objects.hash(key, prefix);
        }
    }
}
