package com.valor.vod.hotkey.common.tool;

import com.valor.vod.hotkey.common.annotation.SPI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.URL;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * @see java.util.ServiceLoader
 * @param <T>
 * @author wukai
 */
@SuppressWarnings("unchecked")
public class ExtensionLoader<T> {

    public static final String SERVICES_FOLDER = "META-INF/services/";
    public static final String HK_FOLDER = "META-INF/services/hk";

    private static final Logger logger = LoggerFactory.getLogger(ExtensionLoader.class);
    private static final ObjectFactory objectFactory = new DefaultObjectFactory();
    /** service loader map */
    private static final Map<Class<?>, ExtensionLoader<?>> EXTENSION_LOADER_MAP =
            new ConcurrentHashMap<>();

    private final Class<?> baseClass;

    private volatile Map<String, Class<?>> cachedClasses;

    private volatile Map<String, Object> cachedActives;

    public ExtensionLoader(Class<?> baseClass) {
        this.baseClass = baseClass;
    }

    public Object getService(String name) {
        if (cachedActives == null) {
            synchronized (this) {
                if (cachedActives == null) {
                    cachedActives = new ConcurrentSkipListMap<>();
                }
            }
        }
        Object obj = cachedActives.get(name);
        if (obj == null) {
            synchronized (this) {
                obj = cachedActives.get(name);
                if (obj == null) {
                    if (cachedClasses == null) {
                        cachedClasses = loadServiceClasses();
                    }
                    Class<?> clazz = cachedClasses.get(name);
                    if (clazz == null) {
                        throw new ServiceConfigurationException(
                                "No such service named:"
                                        + name
                                        + ", baseClass:"
                                        + baseClass.getName());
                    }
                    try {
                        obj = objectFactory.buildObject(clazz);
                        cachedActives.put(name, obj);
                    } catch (Exception e) {
                        throw new ServiceConfigurationException(
                                "Can't initialize service class : " + clazz.getName());
                    }
                }
            }
        }
        return obj;
    }

    void loadClasses() {
        if (cachedClasses == null) {
            synchronized (this) {
                if (cachedClasses == null) {
                    this.cachedClasses = loadServiceClasses();
                }
            }
        }
    }

    private Map<String, Class<?>> loadServiceClasses() {
        Map<String, Class<?>> serviceClasses = new HashMap<String, Class<?>>();
        load(this, serviceClasses, SERVICES_FOLDER);
        load(this, serviceClasses, HK_FOLDER);
        return serviceClasses;
    }

    public Object getAdaptiveService() {
        SPI spi = baseClass.getAnnotation(SPI.class);
        if (spi.value().length() > 0) {
            return this.getService(spi.value());
        }
        Object obj = null;
        for (String name : cachedClasses.keySet()) {
            try {
                obj = getService(name);
            } catch (Exception e) {
                logger.debug(e.getMessage(), e);
            }
        }
        if (obj == null) {
            throw new ServiceConfigurationException(baseClass.getName() + " have no implements");
        }
        return obj;
    }

    @SuppressWarnings("unchecked")
    private static <T> void load(
            ExtensionLoader<T> loader, Map<String, Class<?>> serviceClasses, String pos) {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Load service " + pos + loader.baseClass.getName());
            }
            ClassLoader classLoader = findClassLoader();

            Enumeration<URL> urls = null;
            String fileName = pos + loader.baseClass.getName();

            if (classLoader != null) {
                urls = classLoader.getResources(fileName);
            } else {
                urls = ClassLoader.getSystemResources(fileName);
            }

            if (urls != null) {
                while (urls.hasMoreElements()) {
                    URL url = urls.nextElement();
                    BufferedReader br = new BufferedReader(new InputStreamReader(url.openStream()));
                    try {
                        String line;
                        for (line = br.readLine(); line != null; line = br.readLine()) {
                            line = line.trim();
                            if (line.startsWith("#")) {
                                continue;
                            }
                            String name;
                            String value;
                            if (line.contains("=")) {
                                String[] nameValue = line.split("=");
                                name = nameValue[0].trim();
                                value = nameValue[1].trim();
                            } else {
                                name = line;
                                value = line;
                            }
                            try {
                                Class<?> clazz = ClassUtils.forName(value);
                                serviceClasses.put(name, clazz);
                            } catch (Exception ignored) {
                            }
                        }
                    } finally {
                        try {
                            br.close();
                        } catch (IOException ignored) {
                        }
                    }
                }
            }
        } catch (IOException e) {
            throw new ServiceConfigurationException("load error", e);
        }
    }

    private static ClassLoader findClassLoader() {
        return ClassUtils.getClassLoader(ExtensionLoader.class);
    }

    private static <T> T loadService(Class<T> service, String serviceName, ClassLoader cl) {
        Class<?> c;
        try {
            c = Class.forName(serviceName, false, cl);
        } catch (ClassNotFoundException e) {
            throw new ServiceConfigurationException("class not found", e);
        }
        if (!service.isAssignableFrom(c)) {
            throw new ServiceConfigurationException(
                    serviceName + " not a subtype of " + service.getName());
        }
        try {
            return service.cast(c.newInstance());
        } catch (Exception e) {
            throw new ServiceConfigurationException(serviceName + " initialize fail");
        }
    }

    public static <T> ExtensionLoader<T> getServiceLoader(Class<T> service) {
        checkClassSpi(service);
        ExtensionLoader<T> loader = (ExtensionLoader<T>) EXTENSION_LOADER_MAP.get(service);
        if (loader == null) {
            loader = new ExtensionLoader<T>(service);
            EXTENSION_LOADER_MAP.putIfAbsent(service, loader);
            loader = (ExtensionLoader<T>) EXTENSION_LOADER_MAP.get(service);
        }
        return loader;
    }

    public static <T> T getService(String name, Class<T> service) {
        checkClassSpi(service);
        ExtensionLoader<T> loader = getServiceLoader(service);
        return (T) loader.getService(name);
    }

    public static <T> T getDefaultService(Class<T> service) {
        checkClassSpi(service);
        SPI spi = service.getAnnotation(SPI.class);
        if (spi.value().length() <= 0) {
            throw new ServiceConfigurationException(service.getName() + " has no default service");
        }
        return getService(spi.value(), service);
    }

    private static void checkClassSpi(Class<?> service) {
        if (service == null || !service.isInterface() || service.getAnnotation(SPI.class) == null) {
            throw new ServiceConfigurationException("SPI annotation is missing!");
        }
    }

    /**
     * 获取自适应Service
     *
     * @param service
     * @param <T>
     * @return
     */
    public static <T> T getAdaptiveService(Class<T> service) {
        checkClassSpi(service);
        ExtensionLoader<T> loader = getServiceLoader(service);
        return (T) loader.getAdaptiveService();
    }
}
