package com.valor.vod.hotkey.configcenter.etcd;

import com.google.protobuf.ByteString;
import com.ibm.etcd.api.*;
import com.ibm.etcd.client.KvStoreClient;
import com.ibm.etcd.client.kv.KvClient;
import com.ibm.etcd.client.kv.WatchUpdate;
import com.ibm.etcd.client.lease.LeaseClient;
import com.ibm.etcd.client.lease.PersistentLease;
import com.ibm.etcd.client.lock.LockClient;
import com.valor.vod.hotkey.common.configcenter.AbstractConfigCenter;
import com.valor.vod.hotkey.common.configcenter.model.*;
import com.valor.vod.hotkey.common.tool.CollectionUtil;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * etcd客户端
 *
 * @author wuweifeng wrote on 2019-12-06
 * @version 1.0
 */
public class EtcdConfigCenter extends AbstractConfigCenter implements StreamObserver<WatchUpdate> {

    private static final Logger logger = LoggerFactory.getLogger(EtcdConfigCenter.class);

    private KvClient kvClient;
    private LeaseClient leaseClient;
    private LockClient lockClient;
    private final Map<SubscribeKey, KvClient.Watch> watchMap = new ConcurrentHashMap<>();

    private static final Map<Event.EventType, EventType> EVENT_TYPE_MAP = new HashMap<>();

    static {
        EVENT_TYPE_MAP.put(Event.EventType.PUT, EventType.PUT);
        EVENT_TYPE_MAP.put(Event.EventType.DELETE, EventType.DELETE);
        EVENT_TYPE_MAP.put(Event.EventType.UNRECOGNIZED, EventType.UNRECOGNIZED);
    }

    public EtcdConfigCenter(KvStoreClient kvStoreClient) {
        this.kvClient = kvStoreClient.getKvClient();
        this.leaseClient = kvStoreClient.getLeaseClient();
        this.lockClient = kvStoreClient.getLockClient();
    }

    public LeaseClient getLeaseClient() {
        return leaseClient;
    }

    public void setLeaseClient(LeaseClient leaseClient) {
        this.leaseClient = leaseClient;
    }

    public KvClient getKvClient() {
        return kvClient;
    }

    public void setKvClient(KvClient kvClient) {
        this.kvClient = kvClient;
    }

    public LockClient getLockClient() {
        return lockClient;
    }

    public void setLockClient(LockClient lockClient) {
        this.lockClient = lockClient;
    }

    @Override
    public void put(String key, String value) {
        kvClient.put(ByteString.copyFromUtf8(key), ByteString.copyFromUtf8(value)).sync();
    }

    @Override
    public void add(String key, String value) {
        kvClient.put(ByteString.copyFromUtf8(key), ByteString.copyFromUtf8(value)).sync();
    }

    @Override
    public void put(String key, String value, long leaseId) {
        kvClient.put(ByteString.copyFromUtf8(key), ByteString.copyFromUtf8(value), leaseId).sync();
    }

    @Override
    public void revoke(long leaseId) {
        leaseClient.revoke(leaseId);
    }

    @Override
    public long putAndGrant(String key, String value, long ttl) {
        LeaseGrantResponse lease = leaseClient.grant(ttl).sync();
        put(key, value, lease.getID());
        return lease.getID();
    }

    @Override
    public long setLease(String key, long leaseId) {
        kvClient.setLease(ByteString.copyFromUtf8(key), leaseId);
        return leaseId;
    }

    @Override
    public void delete(String key) {
        kvClient.delete(ByteString.copyFromUtf8(key)).sync();
    }

    @Override
    public String get(String key) {
        RangeResponse rangeResponse = kvClient.get(ByteString.copyFromUtf8(key)).sync();
        List<KeyValue> keyValues = rangeResponse.getKvsList();

        if (CollectionUtil.isEmpty(keyValues)) {
            return null;
        }
        return keyValues.get(0).getValue().toStringUtf8();
    }

    @Override
    public KvData getKvData(String key) {
        RangeResponse rangeResponse = kvClient.get(ByteString.copyFromUtf8(key)).sync();
        List<KeyValue> keyValues = rangeResponse.getKvsList();
        if (CollectionUtil.isEmpty(keyValues)) {
            return null;
        }
        return convert(keyValues.get(0));
    }

    @Override
    public List<KvData> getKvDataPrefix(String key) {
        RangeResponse rangeResponse = kvClient.get(ByteString.copyFromUtf8(key)).asPrefix().sync();
        return rangeResponse.getKvsList().stream().map(this::convert).collect(Collectors.toList());
    }

    @Override
    public long keepAlive(String key, String value, int frequencySecs, int minTtl)
            throws Exception {
        // minTtl秒租期，每frequencySecs秒续约一下
        PersistentLease lease =
                leaseClient
                        .maintain()
                        .leaseId(System.currentTimeMillis())
                        .keepAliveFreq(frequencySecs)
                        .minTtl(minTtl)
                        .start();
        long newId = lease.get(3L, SECONDS);
        put(key, value, newId);
        return newId;
    }

    @Override
    public long buildAliveLease(int frequencySecs, int minTtl) throws Exception {
        PersistentLease lease =
                leaseClient
                        .maintain()
                        .leaseId(System.currentTimeMillis())
                        .keepAliveFreq(frequencySecs)
                        .minTtl(minTtl)
                        .start();

        return lease.get(3L, SECONDS);
    }

    @Override
    public long buildNormalLease(long ttl) {
        LeaseGrantResponse lease = leaseClient.grant(ttl).sync();
        return lease.getID();
    }

    @Override
    public long timeToLive(long leaseId) {
        try {
            return leaseClient.ttl(leaseId).get().getTTL();
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
            return 0L;
        }
    }

    @Override
    protected void doSubscribe(SubscribeKey sk) {
        watchMap.computeIfAbsent(
                sk,
                key -> {
                    KvClient.FluentWatchRequest watchRequest =
                            kvClient.watch(ByteString.copyFromUtf8(sk.getKey()));
                    if (sk.isPrefix()) {
                        watchRequest.asPrefix();
                    }
                    return watchRequest.start(EtcdConfigCenter.this);
                });
    }

    @Override
    protected void doUnsubscribe(SubscribeKey sk) {
        KvClient.Watch watch = watchMap.remove(sk);
        if (watch != null) {
            watch.close();
        }
    }

    @Override
    public void onNext(WatchUpdate watchUpdate) {
        List<Event> events = watchUpdate.getEvents();
        if (events != null && !events.isEmpty()) {
            for (Event event : events) {
                KvData kvData = convert(event.getKv());
                fireDataChange(
                        new DataEvent(
                                kvData.getKey(), kvData, EVENT_TYPE_MAP.get(event.getType())));
            }
        }
    }

    @Override
    public void onError(Throwable throwable) {
        logger.error("Watch error.", throwable);
    }

    @Override
    public void onCompleted() {}

    private KvData convert(KeyValue keyValue) {
        KvData kvData = new KvData();
        kvData.setKey(keyValue.getKey().toStringUtf8());
        kvData.setValue(keyValue.getValue().toStringUtf8());
        kvData.getProperties().put("lease", keyValue.getLease());
        kvData.getProperties().put("modRevision", keyValue.getModRevision());
        return kvData;
    }
}
