package com.valor.vod.api.model.user;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @author Tom Tang
 * @date 2022/11/4
 */
public class LoginIdTypeConverter {

    private static final Map<Integer, ELoginIdType> USER_TYPE_TO_LOGIN_ID_TYPE_MAP =
            new HashMap<Integer, ELoginIdType>() {
                {
                    put(EUserType.MAC.getCode(), ELoginIdType.DEVICE);
                    put(EUserType.ACCOUNT_CARD.getCode(), ELoginIdType.ACCOUNT_CARD);
                    put(EUserType.EMAIL.getCode(), ELoginIdType.EMAIL);
                    put(EUserType.EMAIL_BINDING.getCode(), ELoginIdType.EMAIL);
                    put(EUserType.MOBILE.getCode(), ELoginIdType.MOBILE);
                    put(EUserType.MOBILE_BINDING.getCode(), ELoginIdType.MOBILE);
                    put(EUserType.THIRD_PARTY.getCode(), ELoginIdType.THIRD_PARTY);
                }
            };

    private static final Map<Integer, ELoginIdType> ACCOUNT_LOGIN_TYPE_TO_LOGIN_ID_TYPE_MAP =
            new HashMap<Integer, ELoginIdType>() {
                {
                    put(EAccountLoginType.DEVICE.getCode(), ELoginIdType.DEVICE);
                    put(EAccountLoginType.ACCOUNT_CARD.getCode(), ELoginIdType.ACCOUNT_CARD);
                    put(EAccountLoginType.EMAIL.getCode(), ELoginIdType.EMAIL);
                    put(EAccountLoginType.AC_BINDING_EMAIL.getCode(), ELoginIdType.EMAIL);
                    put(EAccountLoginType.MOBILE.getCode(), ELoginIdType.MOBILE);
                    put(EAccountLoginType.AC_BINDING_MOBILE.getCode(), ELoginIdType.MOBILE);
                    put(EAccountLoginType.THIRD_PARTY.getCode(), ELoginIdType.THIRD_PARTY);
                }
            };

    private static final Map<String, EAccountLoginType>
            LOGIN_ID_TYPE_NAME_TO_ACCOUNT_LOGIN_TYPE_MAP =
                    new HashMap<String, EAccountLoginType>() {
                        {
                            put(ELoginIdType.DEVICE.name(), EAccountLoginType.DEVICE);
                            put(ELoginIdType.ACCOUNT_CARD.name(), EAccountLoginType.ACCOUNT_CARD);
                            put(ELoginIdType.EMAIL.name(), EAccountLoginType.EMAIL);
                            put(ELoginIdType.MOBILE.name(), EAccountLoginType.MOBILE);
                            put(ELoginIdType.THIRD_PARTY.name(), EAccountLoginType.THIRD_PARTY);
                        }
                    };

    public static Optional<ELoginIdType> ofUserType(Integer userType) {
        return Optional.ofNullable(USER_TYPE_TO_LOGIN_ID_TYPE_MAP.get(userType));
    }

    public static Optional<ELoginIdType> ofAccountLoginType(int accountLoginType) {
        return Optional.ofNullable(ACCOUNT_LOGIN_TYPE_TO_LOGIN_ID_TYPE_MAP.get(accountLoginType));
    }

    public static String ofUserTypeToName(Integer userType) {
        return ofUserType(userType).map(Enum::name).orElse(null);
    }

    public static String ofAccountLoginTypeToName(Integer accountLoginType) {
        return ofAccountLoginType(accountLoginType).map(Enum::name).orElse(null);
    }

    public static Optional<EAccountLoginType> toAccountLoginType(String loginIdType) {
        return Optional.ofNullable(LOGIN_ID_TYPE_NAME_TO_ACCOUNT_LOGIN_TYPE_MAP.get(loginIdType));
    }

    public static Integer toAccountLoginTypeCode(String loginIdType) {
        return toAccountLoginType(loginIdType).map(EAccountLoginType::getCode).orElse(null);
    }
}
