package com.valor.vod.cloud.cache;

import com.valor.vod.cloud.condition.CloudRedisClusterConnectionConditon;
import com.valor.vod.cloud.condition.CloudRedisSingleConnectionCondition;
import com.valor.vod.common.cache.redis.FastJsonRedisSerializer;
import common.config.tools.config.ConfigTools3;
import io.lettuce.core.ClientOptions;
import io.lettuce.core.SocketOptions;
import io.lettuce.core.TimeoutOptions;
import io.lettuce.core.resource.ClientResources;
import io.lettuce.core.resource.DefaultClientResources;
import io.lettuce.core.resource.Delay;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisClusterNode;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettucePoolingClientConfiguration;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;
import java.util.concurrent.TimeUnit;


@Configuration
public class CloudCacheRedisConfig {
    private static final Logger log = LoggerFactory.getLogger(CloudCacheRedisConfig.class);

    int dbIndex = ConfigTools3.getInt("mfc.cloud.redis.database", 0);
    String host = ConfigTools3.getString("mfc.cloud.redis.host", "");
    int port = ConfigTools3.getInt("mfc.cloud.redis.port", 0);
    String password = ConfigTools3.getString("mfc.cloud.redis.password", "");

    long connectTimeout = ConfigTools3.getLong("mfc.cloud.redis.connectTimeout", 100L);

    long fixedTimeout = ConfigTools3.getLong("mfc.cloud.redis.fixedTimeout", 5L);

    @Bean(name = "cloudRedisConnectionFactory")
    @Conditional(value = CloudRedisSingleConnectionCondition.class)
    public RedisConnectionFactory getRedisConnectionFactory() {
        return createRedisConnectionFactory(dbIndex, host, port, password, connectTimeout, fixedTimeout);
    }

    @Bean(name = "cloudRedisConnectionFactory")
    @Conditional(value = CloudRedisClusterConnectionConditon.class)
    public RedisConnectionFactory getClusterRedisConnectionFactory() {
        return createRedisClusterConnectionFactory(host, port, password, connectTimeout, fixedTimeout);
    }

    public LettuceConnectionFactory createRedisClusterConnectionFactory(String host, int port, String password, Long connectTimeout, Long fixedTimeout) {
        RedisClusterNode node = new RedisClusterNode(host, port);
        RedisClusterConfiguration config = new RedisClusterConfiguration();
        config.addClusterNode(node);
        config.setPassword(password);

        return new LettuceConnectionFactory(config, configPool(connectTimeout, fixedTimeout));
    }

    private LettuceConnectionFactory createRedisConnectionFactory(int dbIndex, String host, int port, String password, Long connectTimeout, Long fixedTimeout) {
        RedisStandaloneConfiguration redisStandaloneConfiguration = new RedisStandaloneConfiguration();
        redisStandaloneConfiguration.setDatabase(dbIndex);
        redisStandaloneConfiguration.setHostName(host);
        redisStandaloneConfiguration.setPort(port);
        redisStandaloneConfiguration.setPassword(password);

        return new LettuceConnectionFactory(redisStandaloneConfiguration, configPool(connectTimeout, fixedTimeout));
    }

    private LettucePoolingClientConfiguration configPool(Long connectTimeout, Long fixedTimeout) {
        ClientResources resources = DefaultClientResources.builder()
            .reconnectDelay(Delay.fullJitter(
                Duration.ofMillis(100),     // minimum 100 millisecond delay
                Duration.ofSeconds(5),      // maximum 5 second delay
                100, TimeUnit.MILLISECONDS) // 100 millisecond base
            ).build();
        ClientOptions options = ClientOptions.builder()
            .socketOptions(SocketOptions.builder().connectTimeout(Duration.ofMillis(connectTimeout)).build()) // 100 millisecond connection timeout
            .timeoutOptions(TimeoutOptions.builder().fixedTimeout(Duration.ofSeconds(fixedTimeout)).build()) // 5 second command timeout
            .build();
        return LettucePoolingClientConfiguration.builder()
            .poolConfig(getPoolConfig())
            .clientResources(resources)
            .clientOptions(options)
            .build();
    }

    /**
     * 设置连接池属性
     *
     * @return
     */
    public GenericObjectPoolConfig getPoolConfig() {
        int redisPoolMaxIdle = ConfigTools3.getInt("mfc.cloud.redis.pool.max-idle", 8);
        int redisPoolMinIdle = ConfigTools3.getInt("mfc.cloud.redis.pool.min-idle", 0);
        int redisPoolMaxActive = ConfigTools3.getInt("mfc.cloud.redis.pool.max-active", 8);
        int redisPoolMaxWait = ConfigTools3.getInt("mfc.cloud.redis.pool.max-wait", -1);

        GenericObjectPoolConfig poolConfig = new GenericObjectPoolConfig();
        poolConfig.setMaxIdle(redisPoolMaxIdle);
        poolConfig.setMinIdle(redisPoolMinIdle);
        poolConfig.setMaxTotal(redisPoolMaxActive);
        poolConfig.setMaxWaitMillis(redisPoolMaxWait);
        poolConfig.setTestOnBorrow(true);
        return poolConfig;
    }


//    @Bean
//    public RedisCacheManager cacheManager(RedisTemplate redisTemplate) {
//        //获得redis缓存管理类
//        RedisCacheManager redisCacheManager = new RedisCacheManager(redisTemplate);
//        // 开启使用缓存名称做为key前缀(这样所有同名缓存会整理在一起比较容易查找)
//        redisCacheManager.setUsePrefix(true);
//
//        //这里可以设置一个默认的过期时间 单位是秒
//        Long expiredTime = ConfigTools3.getLong("mfc.cache.redis.expire.time", 7200l);
//
//        // 设置缓存的过期时间 单位是秒
//        //  Map<String, Long> expires = new HashMap<>();
//        //  expires.put("com.valor.mfc.vms.user.model.database.ddo.UserInfoDdo", 7200l);
//        //  redisCacheManager.setExpires(expires);
//
//        redisCacheManager.setDefaultExpiration(expiredTime);
//
//        return redisCacheManager;
//    }
//
//    @Bean(name = "cacheKeyGenerator")
//    @Override
//    public KeyGenerator keyGenerator() {
//        return (target, method, params) -> {
//            StringBuilder sb = new StringBuilder();
//            sb.append(target.getClass().getName()).append(".");
//            sb.append(method.getName()).append(".");
//
//            for (Object obj : params) {
//                // 由于参数可能不同, hashCode肯定不一样, 缓存的key也需要不一样
//                sb.append(JSON.toJSONString(obj).hashCode());
//            }
//            return sb.toString();
//        };
//    }

    @Bean(name = "cloudRedisTemplate")
    //@Primary
    public RedisTemplate<String, Object> redisTemplate(@Qualifier("cloudRedisConnectionFactory") RedisConnectionFactory cacheRedisConnectionFactory) {

        // RedisConnectionFactory redisConnectionFactory= createJedisConnectionFactory(dbIndex,host,port,password,timeout);
        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(cacheRedisConnectionFactory);

        FastJsonRedisSerializer<Object> fastJsonRedisSerializer = new FastJsonRedisSerializer<>(Object.class);
        // 全局开启AutoType，不建议使用
        // ParserConfig.getGlobalInstance().setAutoTypeSupport(true);
        // 建议使用这种方式，小范围指定白名单
        //ParserConfig.getGlobalInstance().addAccept("com.mfc.");

        // 设置值（value）的序列化采用FastJsonRedisSerializer。
        redisTemplate.setValueSerializer(fastJsonRedisSerializer);
        redisTemplate.setHashValueSerializer(fastJsonRedisSerializer);
        // 设置键（key）的序列化采用StringRedisSerializer。
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setHashKeySerializer(new StringRedisSerializer());

        redisTemplate.afterPropertiesSet();
        return redisTemplate;
    }


}
