package com.valor.vod.cloud.model.util.stat;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import common.base.tools.stat.MetricTools;
import common.base.tools.sys.SystemTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

/**
 * Created by Frank.Huang on 2016/6/3.
 */
public class TimeStatisticsTools {
    public static final String LINE_SEPARATOR = System.getProperty("line.separator");
    
    private static final Logger logger = LoggerFactory.getLogger(TimeStatisticsTools.class);
    public static final Map<String, StatTimeItem> timeStatMap = Maps.newConcurrentMap();
    public static final Map<String, Long>         timeStartMap = Maps.newConcurrentMap();

    private static boolean report2Remote = false;

    public static void report2Remote(boolean report){
        report2Remote = report;
    }

    public static void addTime(String key, int count, long time) {
        StatTimeItem item = timeStatMap.get(key);
        if (item == null) {
            item = new StatTimeItem();
            item.setKey(key);
        }

        item.addCount(count);
        item.addTime(time);

        timeStatMap.put(key, item);
    }

    public static void startTimer(String key) {
        String timeKey = getStatKey(key);
        timeStartMap.put(timeKey, System.currentTimeMillis());
    }

    public static long stopTimer(String key) {
        return stopTimer(key,true);
    }

    public static long stopTimer(String key,boolean isStat) {
        String timeKey = getStatKey(key);
        Long timeStart = timeStartMap.get(timeKey);
        if (timeStart == null) {
            return 0;
        }

        long time = System.currentTimeMillis() - timeStart;
        timeStartMap.remove(timeKey);

        if (report2Remote){
            MetricTools.logTimeConsumed(key.replace("-","_"),time);
        }
        
        if (isStat){
            addTime(key, 1, time);
        }else {
            logger.info("[RUN_SQL] Time Consuming:[{}ms] sql[{}]",time, key);
        }
        return time;
    }

    private static String getStatKey(String key) {
        return String.valueOf(SystemTools.currentThreadId()) + "-" + key;
    }

    public static String dumpMutilline() {
        StringBuilder sb = new StringBuilder();
        if (!timeStatMap.isEmpty()) {
            List<Map.Entry<String, StatTimeItem>> statList = Lists.newArrayList(timeStatMap.entrySet());
            Collections.sort(statList, new Comparator<Map.Entry<String, StatTimeItem>>() {
                @Override
                public int compare(Map.Entry<String, StatTimeItem> o1, Map.Entry<String, StatTimeItem> o2) {
                    return o1.getKey().compareTo(o2.getKey());
                }
            });

            sb.append(LINE_SEPARATOR);
            sb.append("Time:");
            sb.append(LINE_SEPARATOR);
            statList.forEach(e -> {
                sb.append("\t").append(Strings.padEnd(e.getKey(), 30, ' '));
                sb.append("count:").append(format(e.getValue().getCount()));
                sb.append("avg:").append(format(e.getValue().avg()));
                sb.append("range:").append(e.getValue().rangeStat());
                sb.append(LINE_SEPARATOR);
            });

            timeStatMap.clear();
        }

        return sb.toString();
    }


    private static String format(long v) {
        return Strings.padEnd(String.valueOf(v), 6, ' ');
    }

}
