package com.valor.vod.common.web.aop.processor;

import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.web.aop.annotation.ValidParameter;
import com.valor.vod.common.web.tools.VodException;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.annotation.Order;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;

/**
 * @author Bruce Wu
 * @project vod-common-web
 * @since 2022-11-15
 */
@Aspect
@Order(2)
public class ValidatorAspect {

    private Validator validator;

    public ValidatorAspect(Validator validator) {
        this.validator = validator;
    }

    /** 验证参数是否合法 */
    @Pointcut("@annotation(com.valor.vod.common.web.aop.annotation.ValidParameter)")
    public void validParamPointcut() {}

    @Around(value = "validParamPointcut() && @annotation(validParameter)")
    public Object validation(ProceedingJoinPoint pjp, ValidParameter validParameter)
            throws Throwable {
        Object[] args = pjp.getArgs();
        boolean tx = false;

        String language = null;
        for (Object arg : args) {
            if (arg instanceof HttpServletRequest || arg instanceof HttpServletResponse) {
                continue;
            }
            Object lan;
            try {
                lan = MethodUtils.invokeMethod(arg, "getLanguage");
                if (lan != null && StringUtils.hasText(lan.toString())) {
                    language = lan.toString();
                    break;
                }
            } catch (Throwable ignored) {

            }

            try {
                lan = MethodUtils.invokeMethod(arg, "getLang");
                if (lan != null && StringUtils.hasText(lan.toString())) {
                    language = lan.toString();
                    break;
                }
            } catch (Throwable ignored) {

            }
        }
        if (!StringUtils.hasText(language)) {
            language = "en";
        }

        Locale locale = StringUtils.parseLocale(language);
        LocaleContextHolder.setLocale(locale);
        String errorMessage = "";
        for (Object arg : args) {
            if (arg instanceof HttpServletRequest || arg instanceof HttpServletResponse) {
                continue;
            }
            Set<ConstraintViolation<Object>> set = validator.validate(arg, validParameter.groups());
            if (Objects.nonNull(set) && set.size() > 0) {
                ConstraintViolation<Object> constraintViolation = set.stream().findFirst().get();
                errorMessage = constraintViolation.getMessage();
                tx = true;
                break;
            }
        }
        if (tx) {
            throw VodException.builder()
                    .language(locale.getLanguage())
                    .retCode(HttpCode2.RET_INVALID_PARAM)
                    .errCode(HttpCode2.ERR_PARAM_INVALID)
                    .message(errorMessage)
                    .build();
        }
        return pjp.proceed();
    }
}
