package com.valor.vod.common.web.aop.processor;

import com.valor.vod.api.model.common.response.ResponseStatus;
import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.web.aop.annotation.WebApiCallLiteV1;
import com.valor.vod.common.web.request.BaseRequestArgs;
import com.valor.vod.common.web.request.IRequestBaseArgs;
import common.base.tools.exception.ApiException;
import common.base.tools.statistics.CountStatisticsTools2;
import common.config.tools.config.ConfigTools3;
import common.web.tools.http.model.response.WebApiBaseResponse;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;


/**
 * Created by Frank.Huang on 2016/6/20.
 */
@Service
@Aspect
@Order(1)
public class WebApiCallInterceptorLiteV1 {
    private static final Logger logger = LoggerFactory.getLogger(WebApiCallInterceptorLiteV1.class);

    @Pointcut(value = "@annotation(com.valor.vod.common.web.aop.annotation.WebApiCallLiteV1)")
    public void webApiCall() {
    }

    @Around(value = "com.valor.vod.common.web.aop.processor.WebApiCallInterceptorLiteV1.webApiCall()&&@annotation(wac)")
    public Object apiRun(ProceedingJoinPoint pjp, WebApiCallLiteV1 wac) throws ServletException, IOException, ApiException {
        String apiName = getApiName(pjp);
        //String apiPath = getApiPath(pjp);

        CountStatisticsTools2.addCount(apiName, "REQ", 1);
        HttpServletRequest request = getRequest(pjp);
        HttpServletResponse response = getResponse(pjp);
        //String host = HttpTools.getRemoteHost(request);

        //更新参数
        BaseRequestArgs baseArgs = getArgs(pjp);

        if (baseArgs != null) {
            baseArgs.setArgsFromParameter(request);
        }

        try {

            Object retVal = pjp.proceed();

            boolean isSuccess = isSuccess(retVal);
            if (isSuccess) {
                CountStatisticsTools2.addCount(apiName, "RSP", 1);
                //设置cache control 时间
                if (getCacheControl(wac, apiName) > 0) {
                    getResponse(pjp).setHeader("Cache-Control", "max-age=" + wac.cacheControl());
                }
            } else {
                //如果失败，修改http code为555
                response.setStatus(HttpCode2.SERVER_ERROR);
                CountStatisticsTools2.addCount(apiName, "ERR", 1);
            }

            return retVal;
        } catch (Throwable throwable) {
            if (throwable instanceof ApiException) {
                throw (ApiException) throwable;
            }
            logger.error(String.format("API:[%s]", apiName), throwable);
            throw new ApiException(HttpCode2.RET_SYS_EXCEPTION, HttpCode2.ERR_SYS_EXCEPTION, "Internal error");
        }
    }

    public HttpServletRequest getRequest(JoinPoint jp) {
        RequestAttributes requestAttributes = RequestContextHolder.currentRequestAttributes();
        if (requestAttributes instanceof ServletRequestAttributes) {
            return ((ServletRequestAttributes) requestAttributes).getRequest();
        }
        for (Object arg : jp.getArgs()) {
            if (arg instanceof HttpServletRequest) {
                return (HttpServletRequest) arg;
            }
        }
        return null;
    }

    public HttpServletResponse getResponse(JoinPoint jp) {
        RequestAttributes requestAttributes = RequestContextHolder.currentRequestAttributes();
        if (requestAttributes instanceof ServletRequestAttributes) {
            return ((ServletRequestAttributes) requestAttributes).getResponse();
        }
        for (Object arg : jp.getArgs()) {
            if (arg instanceof HttpServletResponse) {
                return (HttpServletResponse) arg;
            }
        }
        return null;
    }

    public BaseRequestArgs getArgs(JoinPoint jp) {
        for (Object arg : jp.getArgs()) {
            if (arg instanceof BaseRequestArgs) {
                return (BaseRequestArgs) arg;
            }
        }
        return null;
    }

    public String getApiName(JoinPoint jp) {
        String apiName = getRequest(jp).getRequestURI();
        apiName = apiName.substring(1).replace("/", "-");
        return apiName;
    }

    public String getApiPath(JoinPoint jp) {
        return getRequest(jp).getRequestURI();
    }

    private void logRequest(WebApiCallLiteV1 webApiCall,
                            String host,
                            String apiName,
                            IRequestBaseArgs args) {
        if (getLogLevel(webApiCall, apiName, "request") == 0) {
            return;
        }

        logReq(host, apiName, obj2String(args));
    }

    private void logResponse(WebApiCallLiteV1 webApiCall,
                             String host,
                             String apiName,
                             Object result,
                             IRequestBaseArgs iArgs) {
        if (getLogLevel(webApiCall, apiName, "response") == 0) {
            return;
        }

        logResponse(true, host, apiName, result, iArgs);
    }

    private void logResponse(boolean isSuccess, String host, String apiName, Object result, IRequestBaseArgs iArgs) {
        if (isSuccess) {
            logRsp(host, apiName, obj2String(iArgs), obj2String(result));
        } else {
            logErrRsp(host, apiName, obj2String(iArgs), obj2String(result));
        }
    }

    private boolean isSuccess(Object result) {
        if (result instanceof ResponseStatus) {
            return (((ResponseStatus) result).getRetCode() == 0);
        } else if (result instanceof WebApiBaseResponse) {
            return (((WebApiBaseResponse) result).getRetCode() == 0);
        }

        return false;
    }


    private int getMinVersion(WebApiCallLiteV1 webApiCall, String apiName) {
        final String minVerKey = "webapi.version.min";
        final String apiVerKey = String.format("%s.%s", minVerKey, apiName);

        int minVersion = ConfigTools3.getInt(apiVerKey, ConfigTools3.getInt(minVerKey, 0));
        if (minVersion == 0) {
            minVersion = webApiCall.minVersion();
        }

        return minVersion;
    }

    private int getLogLevel(WebApiCallLiteV1 webApiCall, String apiName, String msgType) {
        final String defaultKey = "webapi.log.level";
        final String apiMsgKey = String.format("%s.%s.%s", defaultKey, msgType, apiName);
        final String apiKey = String.format("%s.%s", defaultKey, apiName);

        int value = ConfigTools3.getInt(apiMsgKey, 0);
        if (value != 0) {
            return value;
        }

        value = ConfigTools3.getInt(apiKey, 0);
        if (value != 0) {
            return value;
        }

        value = ConfigTools3.getInt(defaultKey, 0);
        if (value != 0) {
            return value;
        }

        value = webApiCall.logLevel();
        return value;
    }

    private int getCacheControl(WebApiCallLiteV1 webApiCall, String apiName) {
        final String defaultKey = "webapi.cache.control";
        final String apiKey = String.format("%s.%s", defaultKey, apiName);

        int value = ConfigTools3.getInt(apiKey, 0);
        if (value != 0) {
            return value;
        }

        value = ConfigTools3.getInt(defaultKey, 0);
        if (value != 0) {
            return value;
        }

        return webApiCall.cacheControl();
    }

    private String obj2String(Object obj) {
        if (obj == null) {
            return "-";
        } else {
            return obj.toString();
        }
    }

    private void logReq(String host,
                        String apiName,
                        String request) {
        logger.info("[REQ] [HOST]:[{}] [API]:[{}]req[{}]",
            host, apiName, request);
    }

    private void logRsp(String host,
                        String apiName,
                        String request,
                        String response) {
        logger.info("[RSP] [HOST]:[{}] [API]:[{}] Resp[{}] req[{}]",
            host, apiName, response, request);
    }

    private void logErrRsp(String host,
                           String apiName,
                           String request,
                           String response) {
        logger.error("[RSP] [HOST]:[{}] [API]:[{}] Resp[{}] req[{}]",
            host, apiName, response, request);
    }

}
