package com.valor.vod.common.web.filter.sso;


import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.valor.vod.api.common.encrypt.RSABase64Utils;
import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.tools.http.ASKeyConfig;
import com.valor.vod.common.tools.http.HttpConstant;
import com.valor.vod.common.tools.http.HttpParameterTools;
import com.valor.vod.common.tools.http.HttpTools;
import common.config.tools.config.ConfigTools3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;

/**
 * Created by Frank.Huang on 2016/6/25.
 */
public class SSOTokenFilter extends AbstractSSOFilter {
    private static final Logger logger = LoggerFactory.getLogger(SSOTokenFilter.class);

    public SSOTokenFilter() {
        excludedSet.addAll(Splitter.on(",").trimResults().splitToList(getStringTokenFilterExclude()));
    }

    @Override
    public String getToken(HttpServletRequest httpRequest) {
        return HttpParameterTools.getParameter(httpRequest, HttpConstant.HTTP_ARG_TOKEN);
    }

    @Override
    public byte[] decryptToken(String token) {
        try {
            return RSABase64Utils.decryptByPublicKey(token, ASKeyConfig.getAsKey());
        } catch (Exception e) {
            logger.error("{}", e);
            return new byte[0];
        }
    }

    @Override
    public int verifyToken(HeaderMapRequestWrapper httpRequest, Object object) {
        String host = HttpTools.getRemoteHost(httpRequest);
        if (object instanceof TokenV2) {
            TokenV2 token = (TokenV2) object;
            long curMS = System.currentTimeMillis();
            if (curMS >= token.getExpireTimestamp()) {
                logger.error("==========HOST:[{}] MAC:[{}] Token expire[{}-{}-{}]", host, token.getDid(),
                    curMS, token.getExpireTimestamp(), token.getExpireTimestamp());
                return HttpCode2.ERR_AUTH_EXPIRED_TOKEN;
            }

            Long uid = HttpParameterTools.getLongParameter(httpRequest, HttpConstant.HTTP_ARG_UID);
            if (uid != null && uid > 0 && uid.longValue() != token.getDid2()) {
                logger.error("==========HOST:[{}] QueryString[{}]", host, httpRequest.getQueryString());
                logger.error("==========HOST:[{}] uid:[{}] INVALID UID[{}]", host, uid, token.getDid2());
                return HttpCode2.ERR_AUTH_UID_CONFLICT;
            }

            Long accountId = HttpParameterTools.getLongParameter(httpRequest, HttpConstant.HTTP_ARG_ACCOUNT_ID);
            if (accountId != null && accountId > 0 && accountId.longValue() != token.getAccountId()) {
                logger.error("==========HOST:[{}] QueryString[{}]", host, httpRequest.getQueryString());
                logger.error("==========HOST:[{}] uid:[{}] INVALID ACCOUNT ID[{}]", host, uid, token.getAccountId());
                return HttpCode2.ERR_AUTH_ACCOUNT_ID_CONFLICT;
            }

            //只有盒子的时候才校验DID
            if (token.getClientType() == TokenV2.TOKEN_FOR_DEVICE) {
                String did = HttpParameterTools.getParameter(httpRequest, HttpConstant.HTTP_ARG_DID);
                if (Strings.isNullOrEmpty(did) || !did.equalsIgnoreCase(token.getDid())) {
                    logger.error("==========HOST:[{}] MAC:[{}] Invalid Mac[{}]", host, did, token.getDid());
                    return HttpCode2.ERR_AUTH_INVALID_DID;
                }
            }

            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_ACCOUNT_ID, token.getAccountId());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_DID, token.getDid());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_UID, token.getDid2());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_TS, token.getGenerateTimestamp());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_UPROP, token.getUserProps());
            httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_TOKEN, getToken(httpRequest));
            httpRequest.setAttribute(HttpConstant.HTTP_ARG_ACCOUNT_TYPE, token.get(AuthTokenFiled.ACCOUNT_TYPE.name()));
            //兼容老版�
            try {
                httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_ROOT_NODE_ID, token.getRootNodeId());
                httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_PRODUCT_ID, token.getProductId());
            } catch (Exception e) {
                httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_ROOT_NODE_ID, 0L);
                httpRequest.setAttribute(HttpConstant.HTTP_REQ_ATTR_PRODUCT_ID, 0L);
            }

            if (token.getVersion() >= TokenV2.V20180808) {
                httpRequest.setAttribute(HttpConstant.HTTP_ARG_CLIENT_SESSION, token.get(AuthTokenFiled.CLIENT_SESSION.name()));
            }

            return HttpCode2.OK;
        } else {
            logger.error("==========HOST:[{}] Invalid token", host);
            return HttpCode2.ERR_AUTH_INVALID_TOKEN;
        }
    }

    @Override
    public Object getTokenObj(HttpServletRequest httpRequest, byte[] bytes) {
        return TokenV2.valueOf(bytes);
    }

    private String getStringTokenFilterExclude() {
        // 注意：新增exclude通过配置项(vod.token.filter.exclude)增加，不要加到这儿
        String defaultTokenFilterExclude = "/api/uc/quicklogin/v1, /api/user/unLockScreen/v1, /api/user/resetScreenPwd/v1, /api/user/restoreScreenPwd/v1, " +
            "/api/oauth/tokenLogin/v1, /api/user/register/v3, /api/user/register/v2, /api/user/area/list/v1, " +
            "/api/user/verificationCode/send/v1, /api/user/mobile/register/v1, /api/uc/register/v2, /api/uc/login/v2, " +
            "/api/user/charge/rechargeByAgent/v1, /api/user/charge/rechargeByAgent/rollback/v1, /api/notification/list/v1, " +
            "/api/notification/list/banner, /api/notification/getLatestUpdateTime/v1, /api/user/bindEmail/activate/v1, " +
            "/api/user/register/v1, /api/user/getEmail/v1, /api/notification/getTips/v1, /api/ping/*, /api/version/*, " +
            "/api/error/v1, /api/production/rpt/v1, /api/scheduler/migrate/execute, /api/uc/queryLoginMethod/v1, " +
            "/api/uc/offlineSession/v1, /api/uc/admin/queryAccountInfo/v1, /api/user/getDeviceInfo/v1, /api/oauth/login/v1, " +
            "/api/uac/login/v1, /api/uac/codeLogin/v1, /api/uac/activate/v1, /api/uac/activatedStatus/v1, /api/ns/*, " +
            "/api/ping2, /api/uc/login4web/v1, /api/uc/login4app/v1, /api/uc/login4m/v1, /api/uc/register/v1, " +
            "/api/uc/checkUser/v1, /api/uc/activateUser/v1, /api/uc/resetPassword/v1, /api/account/notice/setOnlineLimit/v1, " +
            "/api/uc/resetpassword/v2, /api/uc/mobile/resetpassword/v1, /api/uc/resetPassword/v3, /api/playlist/profileMetas/v1, " +
            "/api/playlist/profilePlaylists/v1, /api/profile/foryou/list/v1, /api/notification/list, /api/notification/listForBanner, " +
            "/api/oauth/accountExpireTs/v1, /api/media/getPublicPlaySource/v1, /api/media/getCloudSummary/v1, " +
            "/api/profile/rating/add/v2, /api/profile/rating/query, /api/open/list/v1, /api/open/batchPlaylist/v1, " +
            "/api/schdule/reload/v1, /api/playlist/sortList/v1, /api/media/playCount/v1";

        return ConfigTools3.getString("vod.token.filter.exclude", defaultTokenFilterExclude);
    }
}
