package com.valor.vod.common.web.tools;

import com.valor.vod.api.model.common.response.ResponseStatus;
import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.tools.http.ErrorMsgBuilder;
import com.valor.vod.common.web.util.TraceUtils;
import common.base.tools.exception.ApiException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;

import java.lang.reflect.UndeclaredThrowableException;

/**
 * 统一异常处理器
 *
 * @author Tom Tang
 * @date 2021/7/27
 * @since 3.0.0
 */
@Slf4j
@ConditionalOnMissingBean(annotation = ExceptionHandler.class)
@ControllerAdvice
public class VodExceptionHandler {

    private static final int VOD_ERROR_STATUS = HttpCode2.SERVER_ERROR;

    @ExceptionHandler(Throwable.class)
    private ResponseEntity<ResponseStatus> handThrowable(WebRequest request, Throwable e) {
        log.error("An internal exception occurred. req:{}.", request, e);

        ResponseStatus errorResponse = new ResponseStatus();
        errorResponse.setStatus(HttpCode2.RET_SYS_EXCEPTION, HttpCode2.ERR_SYS_EXCEPTION, "Internal Server Error.");
        errorResponse.setErrMessage("Internal Server Error.");
        setAndRemoveTraceId(errorResponse);
        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    @ExceptionHandler(IllegalArgumentException.class)
    private ResponseEntity<ResponseStatus> handIllegalArgumentException(IllegalArgumentException e) {
        log.warn("[Bad Request]", e);

        String errMsg = ErrorMsgBuilder.buildErrMsgNotAppendCode(null, null,
            HttpCode2.RET_INVALID_PARAM, HttpCode2.ERR_PARAM_INVALID, e.getMessage());
        ResponseStatus errorResponse = new ResponseStatus();
        errorResponse.setStatus(HttpCode2.RET_INVALID_PARAM, HttpCode2.ERR_PARAM_INVALID, errMsg);
        errorResponse.setErrMessage(errMsg);
        setAndRemoveTraceId(errorResponse);

        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    @ExceptionHandler({VodIllegalArgumentException.class, VodIllegalStateException.class})
    private ResponseEntity<ResponseStatus> handVodIllegalException(VodException e) {
        log.warn("[Bad Request]", e);
        return ResponseEntity.status(VOD_ERROR_STATUS).body(buildResponseStatus(e));
    }

    @ExceptionHandler(VodException.class)
    private ResponseEntity<ResponseStatus> handVodException(VodException e) {
        return ResponseEntity.status(VOD_ERROR_STATUS).body(buildResponseStatus(e));
    }

    private ResponseStatus buildResponseStatus(VodException e) {
        ResponseStatus errorResponse = new ResponseStatus();
        String errorMessage = ErrorMsgBuilder.buildErrMsgNotAppendCode(e.getLanguage(), e.getDeviceId(),
            e.getRetCode(), e.getErrCode(), e.getMessage());
        errorResponse.setStatus(e.getRetCode(), e.getErrCode(), errorMessage);
        errorResponse.setErrMessage(errorMessage);
        setAndRemoveTraceId(errorResponse);

        return errorResponse;
    }

    @ExceptionHandler(UndeclaredThrowableException.class)
    private ResponseEntity<ResponseStatus> handVodException(WebRequest request, UndeclaredThrowableException e) {
        Throwable undeclaredThrowable = e.getUndeclaredThrowable();
        if (undeclaredThrowable instanceof ApiException) {
            return handApiException((ApiException) undeclaredThrowable);
        } else if (undeclaredThrowable instanceof BusinessException) {
            return handBusinessException((BusinessException) undeclaredThrowable);
        } else if (undeclaredThrowable instanceof VodException) {
            return handVodException((VodException) undeclaredThrowable);
        } else {
            return handThrowable(request, undeclaredThrowable);
        }
    }

    @ExceptionHandler(ApiException.class)
    private ResponseEntity<ResponseStatus> handApiException(ApiException e) {
        log.warn("A api exception occurred.", e);

        ResponseStatus errorResponse = new ResponseStatus();
        String errorMessage = ErrorMsgBuilder.buildErrMsgNotAppendCode(null, null,
            e.getRetCode(), e.getErrCode(), "Internal Server Error.");
        errorResponse.setStatus(e.getRetCode(), e.getErrCode(), errorMessage);
        errorResponse.setErrMessage(errorMessage);
        setAndRemoveTraceId(errorResponse);

        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    @ExceptionHandler(BusinessException.class)
    private ResponseEntity<ResponseStatus> handBusinessException(BusinessException e) {
        log.warn("A business exception occurred.", e);

        ResponseStatus errorResponse = new ResponseStatus();
        String errorMessage = ErrorMsgBuilder.buildErrMsgNotAppendCode(null, null,
            e.getRetCode(), e.getErrCode(), e.getMessage());
        errorResponse.setStatus(e.getRetCode(), e.getErrCode(), errorMessage);
        errorResponse.setErrMessage(errorMessage);
        setAndRemoveTraceId(errorResponse);

        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    private void setAndRemoveTraceId(ResponseStatus errorResponse) {
        errorResponse.setTraceId(TraceUtils.getTraceId());
        TraceUtils.removeTraceId();
    }
}
