package com.valor.vod.common.web.crypt.filter;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import common.base.tools.JSON.GsonTypeAdapter;
import common.base.tools.type.CollectionUtils;
import org.apache.http.entity.ContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.*;

public class RequestCryptWrapper extends HttpServletRequestWrapper {
    private static final Logger logger = LoggerFactory.getLogger(RequestCryptWrapper.class);

    private Map<String, String[]> parameterMap = new HashMap<String, String[]>();
    private String bodyStr = "";
    private String originalBodyStr = "";
    //    private Gson gson = new Gson();
    private Gson gson = new GsonBuilder().
        registerTypeAdapter(new TypeToken<Map<String, Object>>() {
        }.getType(), new GsonTypeAdapter())
        .create();

    public RequestCryptWrapper(HttpServletRequest request) {
        super(request);

//        bodyStr = getRequestBody(request);
        if (!Strings.isNullOrEmpty(request.getContentType()) && request.getContentType().toLowerCase().contains("multipart/form-data")) {
            logger.info("ContentType is multipart/form-data,request:[{}]", request.getRequestURI());
        } else {
            bodyStr = getRequestBody(request);
        }

        parameterMap.putAll(request.getParameterMap());
    }

    private String getRequestBody(HttpServletRequest request) {
        boolean isOriginalRequest = !(request instanceof RequestCryptWrapper);
        boolean isForm = (request.getContentType() != null
            && (request.getContentType().startsWith(ContentType.APPLICATION_FORM_URLENCODED.getMimeType())
            || request.getContentType().startsWith(ContentType.MULTIPART_FORM_DATA.getMimeType())));
        if (isOriginalRequest && isForm) {
            return "";
        }

        String requestBody = getRequestBodyString(request);
        if (!Strings.isNullOrEmpty(requestBody)) {
            Map<String, Object> postMap = gson.fromJson(requestBody, Map.class);
            if (!CollectionUtils.isNullOrEmpty(postMap)) {
                postMap.forEach((k, v) -> {
                    addParameter(k, v);
                });
            }
        }
        return Strings.nullToEmpty(requestBody);
    }

    private String getRequestBodyString(HttpServletRequest request) {
        BufferedReader br = null;
        StringBuilder sb = new StringBuilder();
        try {
            br = request.getReader();
            String str;
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            br.close();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (null != br) {
                try {
                    br.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return sb.toString();
    }

    // 重写getParameter，代表参数从当前类中的map获取
    @Override
    public String getParameter(String name) {
        String[] values = parameterMap.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }

    // getParameterValues，代表参数从当前类中的map获取
    @Override
    public String[] getParameterValues(String name) {
        return parameterMap.get(name);
    }

    @Override
    public Map<String, String[]> getParameterMap() {
        return parameterMap;
    }

    @Override
    public Enumeration getParameterNames() {
        return Collections.enumeration(this.parameterMap.keySet());
    }

    //HttpServletRequest 的 getInputStream() 和 getReader() 都只能读取一次，由于 Request Body 是流的形式读取，那么
    //流读了一次就没有了，所以只能被调用一次。
    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream(), getCharacterEncoding()));
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {
        return new ServletInputStreamWrapper((Strings.nullToEmpty(originalBodyStr)).getBytes(Charsets.UTF_8));
    }

    @Override
    public String getHeader(String name) {
        String originalHeader = super.getHeader(name);
        if (name.equalsIgnoreCase("content-type") && !Strings.isNullOrEmpty(this.originalBodyStr)) {
            if (!Strings.isNullOrEmpty(originalHeader)) {
                return originalHeader.replace("application/x-www-form-urlencoded", "application/json");
            }
        }
        return originalHeader;
    }

    @Override
    public Enumeration<String> getHeaders(String name) {
        Enumeration<String> headers = super.getHeaders(name);
        if (name.equalsIgnoreCase("Content-Type") && !Strings.isNullOrEmpty(originalBodyStr)) {
            if (headers.hasMoreElements()) {
                ArrayList<String> headList = Collections.list(headers);
                if (!CollectionUtils.isNullOrEmpty(headList)) {
                    if (headList.contains("application/x-www-form-urlencoded")) {
                        Collections.replaceAll(headList, "application/x-www-form-urlencoded", "application/json");
                        return Collections.enumeration(headList);
                    }
                }
            }
        }
        return headers;
    }

    @Override
    public String getContentType() {
        String originalContentType = getHeader("content-type");
        if (!Strings.isNullOrEmpty(this.originalBodyStr)) {
            if (!Strings.isNullOrEmpty(originalContentType)) {
                return originalContentType.replace("application/x-www-form-urlencoded", "application/json");
            }
        }
        return originalContentType;
    }

    public void addAllParameters(Map<String, Object> otherParams) {// 增加多个参数
        if (!CollectionUtils.isNullOrEmpty(otherParams)) {
            for (Map.Entry<String, Object> entry : otherParams.entrySet()) {
                addParameter(entry.getKey(), entry.getValue());
            }
        }
    }


    public void setOriginalBodyStr(String originalBodyStr) {
        this.originalBodyStr = originalBodyStr;
    }

    public void updateOriginalBodyStrFromBodyStr() {
        if (!Strings.isNullOrEmpty(bodyStr)) {
            this.originalBodyStr = bodyStr;
        }
    }

    public void addParameter(String name, Object value) {// 增加参数
        if (value != null) {
            if (value instanceof String[]) {
                parameterMap.put(name, (String[]) value);
            } else if (value instanceof String) {
                parameterMap.put(name, new String[]{(String) value});
            } else {
                parameterMap.put(name, new String[]{String.valueOf(value)});
            }
        }
    }
}
