package com.valor.vod.common.web.aop.processor;

import com.valor.vod.common.web.aop.annotation.InterfaceLog;

import lombok.extern.slf4j.Slf4j;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;

/**
 * @author Tom Tang
 * @date 2022/11/21
 */
@Slf4j
@Aspect
@Component
public class InterfaceLogAspect {

    @Around(
            value =
                    "@annotation(com.valor.vod.common.web.aop.annotation.InterfaceLog) && @annotation(annotation)")
    public Object log(ProceedingJoinPoint pjp, InterfaceLog annotation) throws Throwable {
        Method method = ((MethodSignature) pjp.getSignature()).getMethod();

        log.info(
                "{}.{} start. {}",
                method.getDeclaringClass().getSimpleName(),
                method.getName(),
                getParamString(method, pjp.getArgs()));

        Object returnValue = null;
        try {
            returnValue = pjp.proceed();
            return returnValue;
        } finally {
            log.info(
                    "{}.{} end. returnValue[{}]",
                    method.getDeclaringClass().getSimpleName(),
                    method.getName(),
                    returnValue);
        }
    }

    private String getParamString(Method method, Object[] args) {
        DefaultParameterNameDiscoverer discoverer = new DefaultParameterNameDiscoverer();
        String[] parameterNames = discoverer.getParameterNames(method);
        if (parameterNames == null || parameterNames.length == 0) {
            return "";
        }

        StringBuilder messageBuilder = new StringBuilder(64);
        for (int i = 0; i < parameterNames.length; i++) {
            messageBuilder
                    .append(' ')
                    .append(parameterNames[i])
                    .append('[')
                    .append(args[i])
                    .append("]")
                    .append(i == parameterNames.length - 1 ? '.' : ',');
        }

        return messageBuilder.toString();
    }
}
