package com.valor.vod.common.web.configuration;

import com.valor.vod.common.web.constant.ErrorCode;
import com.valor.vod.common.web.constant.ProjectModule;
import com.valor.vod.common.web.tools.ErrorCodeException;
import com.valor.vod.common.web.tools.ErrorManager;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.core.env.Environment;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.core.type.classreading.CachingMetadataReaderFactory;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.core.type.filter.AssignableTypeFilter;
import org.springframework.core.type.filter.TypeFilter;
import org.springframework.util.ClassUtils;

import java.io.IOException;

/**
 * 错误码扫描器
 *
 * @author Bruce Wu
 * @since 2023-03-06
 */
public class ClassPathErrorCodeScanner implements EnvironmentAware, ResourceLoaderAware {

    static final String DEFAULT_RESOURCE_PATTERN = "**/*.class";

    private String resourcePattern = DEFAULT_RESOURCE_PATTERN;

    private Environment environment;

    private ResourcePatternResolver resourcePatternResolver;

    private MetadataReaderFactory metadataReaderFactory;

    private final TypeFilter enumFilter;
    private final TypeFilter errorCodeFilter;
    private final TypeFilter projectModuleFilter;

    public ClassPathErrorCodeScanner() {
        errorCodeFilter = new AssignableTypeFilter(ErrorCode.class);
        projectModuleFilter = new AssignableTypeFilter(ProjectModule.class);
        enumFilter = new AssignableTypeFilter(Enum.class);
    }

    @Override
    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourcePatternResolver =
                ResourcePatternUtils.getResourcePatternResolver(resourceLoader);
        this.metadataReaderFactory = new CachingMetadataReaderFactory(resourceLoader);
    }

    @Override
    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }

    public void setResourcePattern(String resourcePattern) {
        this.resourcePattern = resourcePattern;
    }

    public Environment getEnvironment() {
        return environment;
    }

    /**
     * 扫描
     *
     * @param basePackages
     */
    public void scan(String... basePackages) {
        for (String basePackage : basePackages) {
            doScan(basePackage);
        }
    }

    protected void doScan(String basePackage) {
        try {
            String packageSearchPath =
                    ResourcePatternResolver.CLASSPATH_ALL_URL_PREFIX
                            + resolveBasePackage(basePackage)
                            + '/'
                            + this.resourcePattern;
            Resource[] resources = resourcePatternResolver.getResources(packageSearchPath);
            for (Resource resource : resources) {
                try {
                    MetadataReader metadataReader =
                            metadataReaderFactory.getMetadataReader(resource);
                    if (projectModuleFilter.match(metadataReader, metadataReaderFactory)) {
                        registerProjectModule(metadataReader, metadataReaderFactory);
                    } else if (errorCodeFilter.match(metadataReader, metadataReaderFactory)) {
                        registerErrorCode(metadataReader, metadataReaderFactory);
                    }
                } catch (Exception e) {
                    throw new ErrorCodeException("Unexpected failure during register", e);
                }
            }
        } catch (IOException e) {
            throw new ErrorCodeException("I/O failure during classpath scanning", e);
        }
    }

    protected String resolveBasePackage(String basePackage) {
        return ClassUtils.convertClassNameToResourcePath(
                getEnvironment().resolveRequiredPlaceholders(basePackage));
    }

    private void registerProjectModule(
            MetadataReader metadataReader, MetadataReaderFactory metadataReaderFactory)
            throws Exception {
        if (enumFilter.match(metadataReader, metadataReaderFactory)) {
            Class<?> clazz =
                    ClassUtils.forName(
                            metadataReader.getClassMetadata().getClassName(),
                            ClassUtils.getDefaultClassLoader());
            for (Object o : clazz.getEnumConstants()) {
                ErrorManager.register((ProjectModule) o);
            }
        }
    }

    private void registerErrorCode(
            MetadataReader metadataReader, MetadataReaderFactory metadataReaderFactory)
            throws Exception {
        if (enumFilter.match(metadataReader, metadataReaderFactory)) {
            Class<?> clazz =
                    ClassUtils.forName(
                            metadataReader.getClassMetadata().getClassName(),
                            ClassUtils.getDefaultClassLoader());
            for (Object o : clazz.getEnumConstants()) {
                ErrorManager.register((ErrorCode) o);
            }
        }
    }
}
