package com.valor.vod.common.web.configuration;

import com.valor.vod.common.tools.http.HttpTools;
import com.valor.vod.common.web.constant.VodHttpAttr;
import com.valor.vod.common.web.constant.VodHttpHeader;
import com.valor.vod.common.web.tools.VodHttpContext;

import org.apache.skywalking.apm.toolkit.trace.TraceContext;
import org.springframework.http.HttpHeaders;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 请求头提取Interceptor
 *
 * @author Bruce Wu
 * @date 2023-03-15
 */
public class HeaderExtractHandlerInterceptor implements HandlerInterceptor {

    private Map<String, String> extractHeaders;

    public HeaderExtractHandlerInterceptor(Map<String, String> extractHeaders) {
        this.extractHeaders = extractHeaders;
    }

    public HeaderExtractHandlerInterceptor() {
        this(Collections.emptyMap());
    }

    @Override
    public boolean preHandle(
            HttpServletRequest request, HttpServletResponse response, Object handler)
            throws Exception {
        // ThreadLocal缓存全部清除
        VodHttpContext.getContext().removeAll();
        // 提取自定义的header
        if (extractHeaders != null && !extractHeaders.isEmpty()) {
            for (Map.Entry<String, String> en : extractHeaders.entrySet()) {
                String val = request.getHeader(en.getKey());
                if (val != null) {
                    VodHttpContext.getContext().set(VodHttpAttr.valueOf(en.getValue()), val);
                }
            }
        }

        for (VodHttpHeader header : VodHttpHeader.values()) {
            Object val = request.getHeader(header.getHeaderName());
            if (val == null) {
                val = request.getAttribute(header.getAttr().getName());
            }
            if (val != null) {
                VodHttpContext.getContext().set(header.getAttr(), val);
            }
        }

        // 处理Language，没传默认en
        String acceptLanguage = request.getHeader(HttpHeaders.ACCEPT_LANGUAGE);
        Locale locale = Locale.ENGLISH;
        if (StringUtils.hasText(acceptLanguage)) {
            HttpHeaders httpHeaders = new HttpHeaders();
            httpHeaders.add(HttpHeaders.ACCEPT_LANGUAGE, acceptLanguage);
            List<Locale> locales = httpHeaders.getAcceptLanguageAsLocales();
            if (!CollectionUtils.isEmpty(locales)) {
                locale = locales.get(0);
            }
        }
        VodHttpContext.getContext().set(VodHttpAttr.LANGUAGE, locale.getLanguage());

        String clientIp = HttpTools.getRemoteHost(request);
        VodHttpContext.getContext().set(VodHttpAttr.CLIENT_IP, clientIp);

        return true;
    }

    @Override
    public void postHandle(
            HttpServletRequest request,
            HttpServletResponse response,
            Object handler,
            ModelAndView modelAndView)
            throws Exception {
        String cryptVerHeaderName = "Vod-CryptVer";
        // 追加个加解密版本号
        String cryptVer = request.getHeader(cryptVerHeaderName);
        if (StringUtils.hasText(cryptVer)) {
            response.setHeader(cryptVerHeaderName, cryptVer);
        }
        // 响应头增加 Skywalking TraceId
        response.setHeader("X-Sw-Trace-Id", TraceContext.traceId());
        // ThreadLocal缓存全部清除
        VodHttpContext.getContext().removeAll();
    }
}
