package com.valor.vod.common.web.configuration;

import com.valor.vod.common.web.aop.processor.RequestClientAspect;
import com.valor.vod.common.web.aop.processor.ValidatorAspect;
import com.valor.vod.common.web.aop.processor.VerifySDKSessionInterceptor;
import com.valor.vod.common.web.aop.processor.VerifySessionInterceptor;
import com.valor.vod.common.web.http.CommonServiceApi;

import common.config.tools.config.ConfigTools3;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.AbstractMessageSource;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.Validator;
import org.springframework.validation.beanvalidation.LocalValidatorFactoryBean;
import org.springframework.web.servlet.config.annotation.EnableWebMvc;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import java.text.MessageFormat;
import java.util.List;
import java.util.Locale;

import javax.validation.executable.ExecutableValidator;

/**
 * vod common configuration
 *
 * @author Bruce Wu
 * @since 2023-03-05
 */
@Configuration
@EnableConfigurationProperties(VodCommonWebProperties.class)
@ComponentScan(basePackages = "com.valor.vod.common.web")
public class VodCommonWebConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(VodCommonWebConfiguration.class);

    /**
     * @date 2023-03-30
     */
    public static class ConfigMessageSource extends AbstractMessageSource {
        @Override
        protected MessageFormat resolveCode(String code, Locale locale) {
            // 1.ConfigTool
            String cfgKey = locale.getLanguage() + "." + code;
            String format = ConfigTools3.getString(cfgKey);
            return new MessageFormat(format);
        }
    }

    @Configuration
    @ConditionalOnClass(ExecutableValidator.class)
    @ConditionalOnResource(
            resources = "classpath:META-INF/services/javax.validation.spi.ValidationProvider")
    @AutoConfigureBefore(VodCommonWebMvcConfig.class)
    public static class VodCommonWebValidatorConfiguration {

        @Bean
        public LocalValidatorFactoryBean vcmValidator() {
            LocalValidatorFactoryBean localValidatorFactoryBean = new LocalValidatorFactoryBean();
            localValidatorFactoryBean
                    .getValidationPropertyMap()
                    .put("hibernate.validator.fail_fast", "true");
            localValidatorFactoryBean.setValidationMessageSource(vcmConfigMessageSource());
            return localValidatorFactoryBean;
        }

        @Bean
        @ConditionalOnMissingBean(ValidatorAspect.class)
        @ConditionalOnProperty(
                prefix = "vod.common.web.valid-parameter",
                value = "enabled",
                havingValue = "true")
        public ValidatorAspect vcmValidatorAspect() {
            return new ValidatorAspect(vcmValidator());
        }

        @Bean
        public ConfigMessageSource vcmConfigMessageSource() {
            return new ConfigMessageSource();
        }
    }

    @Configuration
    @ConditionalOnProperty(
            prefix = "vod.common.web.request-client",
            value = "enabled",
            havingValue = "true")
    public static class VodCommonWebRequestClientConfiguration {

        @Bean
        @ConditionalOnMissingBean(RequestClientAspect.class)
        public RequestClientAspect vcmRequestClientAspect() {
            return new RequestClientAspect();
        }
    }

    @Configuration
    @ConditionalOnMissingBean({ErrorCodeScannerConfigurer.class})
    @AutoConfigureAfter
    public static class ErrorCodeScannerRegistrarNotFoundConfiguration
            implements InitializingBean, ApplicationContextAware {

        private ApplicationContext applicationContext;

        @Override
        public void setApplicationContext(ApplicationContext applicationContext)
                throws BeansException {
            this.applicationContext = applicationContext;
        }

        @Override
        public void afterPropertiesSet() {
            logger.debug(
                    "Not found configuration for registering error code bean using @ErrorCodeScan");
            if (!AutoConfigurationPackages.has(this.applicationContext)) {
                return;
            }
            List<String> packages = AutoConfigurationPackages.get(this.applicationContext);
            if (CollectionUtils.isEmpty(packages)) {
                return;
            }
            ClassPathErrorCodeScanner scanner = new ClassPathErrorCodeScanner();
            scanner.setResourceLoader(applicationContext);
            scanner.setEnvironment(applicationContext.getEnvironment());
            scanner.scan(packages.toArray(new String[] {}));
        }
    }

    @Configuration
    @EnableWebMvc
    @EnableConfigurationProperties(VodCommonWebProperties.class)
    @AutoConfigureAfter(VodCommonWebValidatorConfiguration.class)
    public static class VodCommonWebMvcConfig implements WebMvcConfigurer {

        LocalValidatorFactoryBean vcmValidator;
        VodCommonWebProperties properties;

        @Autowired
        public void setProperties(VodCommonWebProperties properties) {
            this.properties = properties;
        }

        @Autowired(required = false)
        @Qualifier("vcmValidator")
        public void setVcmValidator(LocalValidatorFactoryBean vcmValidator) {
            this.vcmValidator = vcmValidator;
        }

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            registry.addInterceptor(
                    new HeaderExtractHandlerInterceptor(properties.getExtractHeaders()));
        }

        @Override
        public Validator getValidator() {
            return vcmValidator;
        }
    }

    @Configuration
    @ConditionalOnProperty(
            prefix = "vod.common.web.verify-session",
            value = "enabled",
            havingValue = "true",
            matchIfMissing = true)
    public static class VodCommonWebVerifySessionConfiguration {

        @Bean
        @ConditionalOnMissingBean(CommonServiceApi.class)
        public CommonServiceApi commonServiceApi() {
            return new CommonServiceApi();
        }

        @Bean
        @ConditionalOnMissingBean(VerifySessionInterceptor.class)
        public VerifySessionInterceptor verifySessionInterceptor() {
            return new VerifySessionInterceptor();
        }

        @Bean
        @ConditionalOnMissingBean(VerifySDKSessionInterceptor.class)
        public VerifySDKSessionInterceptor verifySDKSessionInterceptor() {
            return new VerifySDKSessionInterceptor();
        }
    }
}
