package com.valor.vod.common.web.crypt.filter;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.valor.vod.common.tools.type.CollectionUtils;

import common.base.tools.JSON.GsonTypeAdapter;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.*;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;

/**
 * 加密请求包装
 *
 * @author Frank.Huang
 * @since 2023-03-13
 */
public class RequestCryptWrapper extends HttpServletRequestWrapper {
    private static final Logger logger = LoggerFactory.getLogger(RequestCryptWrapper.class);

    private static final String CONTENT_TYPE = "content-type";
    private static final String APPLICATION_FORM_URLENCODED = "application/x-www-form-urlencoded";
    private static final String MULTIPART_FORM_DATA = "multipart/form-data";
    private static final String APPLICATION_JSON = "application/json";

    private final Map<String, String[]> parameterMap = new HashMap<>();
    private String bodyStr = "";
    private String originalBodyStr = "";
    private static Gson gson =
            new GsonBuilder()
                    .registerTypeAdapter(
                            new TypeToken<Map<String, Object>>() {}.getType(),
                            new GsonTypeAdapter())
                    .create();

    public RequestCryptWrapper(HttpServletRequest request) {
        super(request);
        if (!Strings.isNullOrEmpty(request.getContentType())
                && request.getContentType().toLowerCase().contains(MULTIPART_FORM_DATA)) {
            logger.info("ContentType is multipart/form-data,request:[{}]", request.getRequestURI());
        } else {
            bodyStr = getRequestBody(request);
        }
        parameterMap.putAll(request.getParameterMap());
    }

    private String getRequestBody(HttpServletRequest request) {
        boolean isOriginalRequest = !(request instanceof RequestCryptWrapper);
        String contentType = request.getContentType();
        boolean isForm =
                (contentType != null
                        && (contentType.startsWith(APPLICATION_FORM_URLENCODED)
                                || contentType.startsWith(MULTIPART_FORM_DATA)));
        if (isOriginalRequest && isForm) {
            return "";
        }

        String requestBody = getRequestBodyString(request);
        if (!Strings.isNullOrEmpty(requestBody)) {
            Map<String, Object> postMap = gson.fromJson(requestBody, Map.class);
            if (!CollectionUtils.isNullOrEmpty(postMap)) {
                postMap.forEach(this::addParameter);
            }
        }
        return Strings.nullToEmpty(requestBody);
    }

    private String getRequestBodyString(HttpServletRequest request) {
        BufferedReader br = null;
        StringBuilder sb = new StringBuilder();
        try {
            br = request.getReader();
            String str;
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            br.close();
        } catch (IOException e) {
            logger.warn("Io exception found", e);
        } finally {
            if (null != br) {
                try {
                    br.close();
                } catch (IOException ignored) {
                }
            }
        }
        return sb.toString();
    }

    /**
     * 重写getParameter，代表参数从当前类中的map获取
     *
     * @param name
     * @return
     */
    @Override
    public String getParameter(String name) {
        String[] values = parameterMap.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }

    /**
     * getParameterValues，代表参数从当前类中的map获取
     *
     * @param name
     * @return
     */
    @Override
    public String[] getParameterValues(String name) {
        return parameterMap.get(name);
    }

    @Override
    public Map<String, String[]> getParameterMap() {
        return parameterMap;
    }

    @Override
    public Enumeration<String> getParameterNames() {
        return Collections.enumeration(this.parameterMap.keySet());
    }

    /**
     * HttpServletRequest 的 getInputStream() 和 getReader() 都只能读取一次，由于 Request Body
     * 是流的形式读取，那么流读了一次就没有了，所以只能被调用一次。
     *
     * @return BufferedReader
     */
    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream(), getCharacterEncoding()));
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {
        return new ServletInputStreamWrapper(
                (Strings.nullToEmpty(originalBodyStr)).getBytes(Charsets.UTF_8));
    }

    @Override
    public String getHeader(String name) {
        String originalHeader = super.getHeader(name);
        if (CONTENT_TYPE.equalsIgnoreCase(name) && !Strings.isNullOrEmpty(this.originalBodyStr)) {
            if (!Strings.isNullOrEmpty(originalHeader)) {
                return originalHeader.replace(APPLICATION_FORM_URLENCODED, APPLICATION_JSON);
            }
        }
        return originalHeader;
    }

    @Override
    public Enumeration<String> getHeaders(String name) {
        Enumeration<String> headers = super.getHeaders(name);
        if (!headers.hasMoreElements()) {
            return headers;
        }
        if (CONTENT_TYPE.equalsIgnoreCase(name) && !Strings.isNullOrEmpty(originalBodyStr)) {
            List<String> headList = Collections.list(headers);
            CollectionUtils.replaceAll(
                    headList, e -> e.replace(APPLICATION_FORM_URLENCODED, APPLICATION_JSON));
            return Collections.enumeration(headList);
        }
        return headers;
    }

    @Override
    public String getContentType() {
        String contentType = getHeader(CONTENT_TYPE);
        if (!Strings.isNullOrEmpty(this.originalBodyStr) && !Strings.isNullOrEmpty(contentType)) {
            return contentType.replace(APPLICATION_FORM_URLENCODED, APPLICATION_JSON);
        }
        return contentType;
    }

    public void addAllParameters(Map<String, Object> otherParams) {
        if (!CollectionUtils.isNullOrEmpty(otherParams)) {
            for (Map.Entry<String, Object> entry : otherParams.entrySet()) {
                addParameter(entry.getKey(), entry.getValue());
            }
        }
    }

    public void setOriginalBodyStr(String originalBodyStr) {
        this.originalBodyStr = originalBodyStr;
    }

    public void updateOriginalBodyStrFromBodyStr() {
        if (!Strings.isNullOrEmpty(bodyStr)) {
            this.originalBodyStr = bodyStr;
        }
    }

    public void addParameter(String name, Object value) {
        if (value != null) {
            if (value instanceof String[]) {
                parameterMap.put(name, (String[]) value);
            } else if (value instanceof String) {
                parameterMap.put(name, new String[] {(String) value});
            } else {
                parameterMap.put(name, new String[] {String.valueOf(value)});
            }
        }
    }
}
