package com.valor.vod.common.web.filter.sso;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.hash.Hashing;
import com.valor.vod.api.common.client.IStringSerializable;

import java.io.UnsupportedEncodingException;
import java.util.List;

public class TokenV1 implements IStringSerializable {
    public static final int TOKEN_FOR_WEB = 1;
    public static final int TOKEN_FOR_DEVICE = 2;
    public static final int TOKEN_FOR_ACCOUNT = 3;

    private static final String FIELD_SEPARATOR = "#!";
    private static final int VERSION_20170905 = 20170905;
    private static final int VERSION_20170905_FIELD_SIZE = 11;

    private static final int INDEX_VERSION = 0;
    private static final int INDEX_SIGN = 1;
    private static final int INDEX_GENERATE_TIMESTAMP = 2;
    private static final int INDEX_EXPIRE_TIMESTAMP = 3;
    private static final int INDEX_CLIENT_TYPE = 4;
    private static final int INDEX_LOGIN_TYPE = 5;
    private static final int INDEX_ECHOSTR = 6;
    private static final int INDEX_ACCOUNT_ID = 7;
    private static final int INDEX_USERPROPS = 8;
    private static final int INDEX_DID = 9;
    private static final int INDEX_DID2 = 10;

    private int version = VERSION_20170905; // 版本号
    private Long sign = 0L; // 签名
    private Long generateTimestamp = System.currentTimeMillis(); // token 生成时间
    private Long expireTimestamp = System.currentTimeMillis(); // token失效时间
    private int clientType = 0; //
    private int loginType = 0; // 1 : 设备登陆  2: user登录
    private String echoStr = ""; // 回显字符串
    private Long accountId = 0L; // 用户id
    private Long userProps = 0L; // 用户属性
    private String did = ""; // 设备id(mac)
    private Long did2 = 0L; // 设备id(设备内部UID)

    public static TokenV1 valueOf(String value) {
        if (Strings.isNullOrEmpty(value)) {
            throw new IllegalArgumentException("[TOKEN2] Empty.");
        }

        List<String> valueList = Splitter.on(FIELD_SEPARATOR).splitToList(value);
        if (valueList.size() < 2) {
            throw new IllegalArgumentException("[TOKEN2] Invalid Content");
        }
        TokenV1 token = new TokenV1();
        token.version = Integer.valueOf(valueList.get(INDEX_VERSION));
        if (token.version == VERSION_20170905) {
            if (valueList.size() != VERSION_20170905_FIELD_SIZE) {
                throw new IllegalArgumentException("[TOKEN2] Invalid Content 2");
            }
        }

        token.sign = Long.valueOf(valueList.get(INDEX_SIGN));
        token.generateTimestamp = Long.valueOf(valueList.get(INDEX_GENERATE_TIMESTAMP));
        token.expireTimestamp = Long.valueOf(valueList.get(INDEX_EXPIRE_TIMESTAMP));

        token.clientType = Integer.valueOf(valueList.get(INDEX_CLIENT_TYPE));
        token.loginType = Integer.valueOf(valueList.get(INDEX_LOGIN_TYPE));
        token.echoStr = valueList.get(INDEX_ECHOSTR);

        token.accountId = Long.valueOf(valueList.get(INDEX_ACCOUNT_ID));
        token.userProps = Long.valueOf(valueList.get(INDEX_USERPROPS));
        token.did = valueList.get(INDEX_DID);
        token.did2 = Long.valueOf(valueList.get(INDEX_DID2));

        if (token.sign != token.sign()) {
            throw new IllegalArgumentException("[TOKEN2] Invalid SignTools");
        }
        return token;
    }

    public static TokenV1 valueOf(byte[] bytes) {
        return valueOf(new String(bytes));
    }

    private String getSignStr() {
        return Joiner.on(FIELD_SEPARATOR)
                .join(
                        version,
                        generateTimestamp,
                        expireTimestamp,
                        clientType,
                        loginType,
                        echoStr,
                        accountId,
                        userProps,
                        did,
                        did2);
    }

    public long sign() {
        return Hashing.sha256().hashString(getSignStr(), Charsets.UTF_8).padToLong();
    }

    @Override
    public String toSerializeString() {
        sign = sign();
        return Joiner.on(FIELD_SEPARATOR)
                .useForNull("0")
                .join(
                        version,
                        sign,
                        generateTimestamp,
                        expireTimestamp,
                        clientType,
                        loginType,
                        echoStr,
                        accountId,
                        userProps,
                        did,
                        did2);
    }

    @Override
    public byte[] toSerializeBytes() {
        try {
            return toSerializeString().getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return new byte[0];
        }
    }

    public long getSign() {
        return sign;
    }

    public void setSign(long sign) {
        this.sign = sign;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    public Long getGenerateTimestamp() {
        return generateTimestamp;
    }

    public void setGenerateTimestamp(Long generateTimestamp) {
        this.generateTimestamp = generateTimestamp;
    }

    public Long getExpireTimestamp() {
        return expireTimestamp;
    }

    public void setExpireTimestamp(Long expireTimestamp) {
        this.expireTimestamp = expireTimestamp;
    }

    public int getClientType() {
        return clientType;
    }

    public void setClientType(int clientType) {
        this.clientType = clientType;
    }

    public String getEchoStr() {
        return echoStr;
    }

    public void setEchoStr(String echoStr) {
        this.echoStr = echoStr;
    }

    public int getLoginType() {
        return loginType;
    }

    public void setLoginType(int loginType) {
        this.loginType = loginType;
    }

    public Long getAccountId() {
        return accountId;
    }

    public void setAccountId(Long accountId) {
        this.accountId = accountId;
    }

    public Long getUserProps() {
        return userProps;
    }

    public void setUserProps(Long userProps) {
        this.userProps = userProps;
    }

    public String getDid() {
        return did;
    }

    public void setDid(String did) {
        this.did = did;
    }

    public long getDid2() {
        return did2;
    }

    public void setDid2(Long did2) {
        this.did2 = did2;
    }

    public void setDid2(long did2) {
        this.did2 = did2;
    }
}
