package com.valor.vod.common.web.tools;

import com.valor.vod.common.web.constant.ErrorCode;
import com.valor.vod.common.web.constant.ProjectModule;
import com.valor.vod.common.web.constant.SystemErrorCodes;
import com.valor.vod.common.web.constant.SystemProjectModules;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 错误码管理器
 *
 * @author Bruce Wu
 * @since 2023-03-03
 */
public class ErrorManager {

    /** Project module map */
    private static final Map<String, ProjectModule> PROJECT_MODULE_MAP = new ConcurrentHashMap<>();
    /** Error code map */
    private static final Map<String, ErrorCode> ERROR_CODE_MAP = new ConcurrentHashMap<>();

    private static final int MIN_HTTP_CODE = 100;
    private static final int MAX_HTTP_CODE = 599;

    static {
        EnumSet.allOf(SystemProjectModules.class).forEach(ErrorManager::register);
        EnumSet.allOf(SystemErrorCodes.class).forEach(ErrorManager::register);
    }

    /**
     * ProjectModule注册，重复检测
     *
     * @param projectModule
     */
    public static void register(ProjectModule projectModule) {
        if (projectModule == null) {
            throw new ErrorCodeException("ProjectModule is null");
        }
        String projectCode = projectModule.getProjectCode();
        String moduleCode = projectModule.getModuleCode();

        if (!SystemProjectModules.contains(projectModule)) {
            // 非 SystemProjectModules 校验参数
            if (projectCode == null || (projectCode = projectCode.trim()).isEmpty()) {
                throw new ErrorCodeException(
                        String.format("ProjectModule:[%s], projectCode is empty", projectModule));
            }
            if (moduleCode == null || (moduleCode = moduleCode.trim()).isEmpty()) {
                throw new ErrorCodeException(
                        String.format("ProjectModule:[%s], moduleCode is empty", projectModule));
            }
        }

        PROJECT_MODULE_MAP.compute(
                getRespCode(projectCode, moduleCode, null),
                (s, e) -> {
                    if (e != null && !Objects.equals(e, projectModule)) {
                        throw new ErrorCodeException(
                                String.format(
                                        "ProjectModule:[%s] in conflict with [%s]",
                                        projectModule, e));
                    }
                    return projectModule;
                });
    }

    /**
     * 错误码注册，重复检测
     *
     * @param errorCode
     */
    public static void register(ErrorCode errorCode) {
        if (errorCode == null) {
            throw new ErrorCodeException("ErrorCode is null");
        }
        ProjectModule projectModule = errorCode.getProjectModule();
        if (projectModule == null) {
            throw new ErrorCodeException(
                    String.format("ErrorCode:[%s], projectModule is null", errorCode));
        }
        String projectCode = projectModule.getProjectCode();
        String moduleCode = projectModule.getModuleCode();

        if (!SystemProjectModules.contains(projectModule)) {
            // 非 SystemProjectModules 校验参数并注册
            if (null == getProjectModule(projectCode, moduleCode)) {
                register(projectModule);
            }
        }

        String code = errorCode.getCode();
        if (!SystemErrorCodes.contains(errorCode)) {
            // 非 SystemErrorCodes 校验参数
            if (code == null || (code = code.trim()).isEmpty()) {
                throw new ErrorCodeException(
                        String.format("ErrorCode:[%s], code is empty", errorCode));
            }
        }

        // 校验http状态码 100-599范围内
        if (errorCode.getHttpCode() < MIN_HTTP_CODE || errorCode.getHttpCode() > MAX_HTTP_CODE) {
            throw new ErrorCodeException(
                    String.format("ErrorCode:[%s], illegal http code", errorCode));
        }

        ERROR_CODE_MAP.compute(
                getRespCode(projectCode, moduleCode, code),
                (s, e) -> {
                    if (e != null && !Objects.equals(e, errorCode)) {
                        throw new ErrorCodeException(
                                String.format(
                                        "ErrorCode:[%s] in conflict with [%s]", errorCode, e));
                    }
                    return errorCode;
                });
    }

    /**
     * 根据projectCode和moduleCode查找ProjectModule
     *
     * @param projectCode
     * @param moduleCode
     * @return ProjectModule
     */
    public static ProjectModule getProjectModule(String projectCode, String moduleCode) {
        if (projectCode == null || (projectCode = projectCode.trim()).isEmpty()) {
            throw new ErrorCodeException("ProjectCode is empty");
        }
        if (moduleCode == null || (moduleCode = moduleCode.trim()).isEmpty()) {
            throw new ErrorCodeException("ModuleCode is empty");
        }
        return PROJECT_MODULE_MAP.get(getRespCode(projectCode, moduleCode, null));
    }

    /**
     * 获取错误码
     *
     * @param projectCode
     * @param moduleCode
     * @param code
     * @return
     */
    public static ErrorCode getErrorCode(String projectCode, String moduleCode, String code) {
        ProjectModule projectModule = getProjectModule(projectCode, moduleCode);
        if (projectModule == null) {
            return null;
        }
        return ERROR_CODE_MAP.get(getRespCode(projectCode, moduleCode, code));
    }

    /**
     * 生成返回错误码
     *
     * @param errorCode
     * @return
     */
    public static String getRespCode(ErrorCode errorCode) {
        ProjectModule pm = errorCode.getProjectModule();
        return getRespCode(pm.getProjectCode(), pm.getModuleCode(), errorCode.getCode());
    }

    /**
     * 生成返回错误码
     *
     * @param projectCode
     * @param moduleCode
     * @param code
     * @return
     */
    public static String getRespCode(String projectCode, String moduleCode, String code) {
        return Stream.of(projectCode, moduleCode, code)
                .filter(Objects::nonNull)
                .map(String::trim)
                .filter(e -> !e.isEmpty())
                .collect(Collectors.joining("."));
    }

    public static List<ErrorCode> getAllErrorCodes() {
        return new ArrayList<>(ERROR_CODE_MAP.values());
    }
}
