package com.valor.vod.common.web.tools;

import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;

import org.springframework.http.HttpStatus;

/**
 * VOD异常类
 *
 * @author Tom Tang
 * @date 2021/7/27
 * @since 3.0.0
 */
@Getter
public class VodException extends RuntimeException {
    /** http 状态 */
    private HttpStatus httpStatus;
    /** 语言 */
    private String language = "en";
    /** 设备ID */
    private String deviceId;
    /** 返回码 */
    private final int retCode;
    /** 错误码 */
    private final int errCode;
    /** 是否使用默认的错误信息 */
    private boolean useDefaultMsg;

    public VodException(int retCode, int errCode) {
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(int retCode, int errCode, String errorMessage) {
        super(errorMessage);
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(int retCode, int errCode, Throwable cause) {
        super(cause);
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(int retCode, int errCode, String errorMessage, Throwable cause) {
        super(errorMessage, cause);
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(HttpStatus httpStatus, int retCode, int errCode) {
        this.httpStatus = httpStatus;
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(HttpStatus httpStatus, int retCode, int errCode, String errorMessage) {
        super(errorMessage);
        this.httpStatus = httpStatus;
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(HttpStatus httpStatus, int retCode, int errCode, Throwable cause) {
        super(cause);
        this.httpStatus = httpStatus;
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(
            HttpStatus httpStatus, int retCode, int errCode, String errorMessage, Throwable cause) {
        this(httpStatus, null, null, retCode, errCode, errorMessage, cause);
    }

    public VodException(
            HttpStatus httpStatus, String language, int retCode, int errCode, String errorMessage) {
        this(httpStatus, language, null, retCode, errCode, errorMessage);
    }

    public VodException(
            HttpStatus httpStatus,
            String language,
            String deviceId,
            int retCode,
            int errCode,
            String errorMessage) {
        super(errorMessage);
        this.httpStatus = httpStatus;
        this.language = language;
        this.deviceId = deviceId;
        this.retCode = retCode;
        this.errCode = errCode;
    }

    public VodException(
            HttpStatus httpStatus,
            String language,
            int retCode,
            int errCode,
            String errorMessage,
            Throwable cause) {
        this(httpStatus, language, null, retCode, errCode, errorMessage, cause);
    }

    public VodException(
            HttpStatus httpStatus,
            String language,
            String deviceId,
            int retCode,
            int errCode,
            String errorMessage,
            Throwable cause) {
        super(errorMessage, cause);
        this.httpStatus = httpStatus;
        this.language = language;
        this.deviceId = deviceId;
        this.retCode = retCode;
        this.errCode = errCode;
    }

    /**
     * Build message
     *
     * @return
     */
    public String buildMessage() {
        if (useDefaultMsg) {
            return ErrorMsgBuilder.spliceErrMsg(deviceId, retCode, errCode, getMessage());
        } else {
            return ErrorMsgBuilder.buildErrMsg(language, deviceId, retCode, errCode, getMessage());
        }
    }

    // ================ builder =================

    /**
     * builder
     *
     * @return builder
     */
    public static VodExceptionBuilder builder() {
        return new VodExceptionBuilder();
    }

    @Setter
    @Accessors(fluent = true, chain = true)
    public static class VodExceptionBuilder {

        private HttpStatus httpStatus;
        private String language = "en";
        private String deviceId;
        private int retCode;
        private int errCode;
        private boolean useDefaultMsg;
        private String message;
        private Throwable cause;

        public VodException build() {
            VodException vodException;
            if (cause == null) {
                vodException = new VodException(retCode, errCode, message);
            } else {
                vodException = new VodException(retCode, errCode, message, cause);
            }
            vodException.httpStatus = httpStatus;
            vodException.language = language;
            vodException.deviceId = deviceId;
            vodException.useDefaultMsg = useDefaultMsg;
            return vodException;
        }
    }
}
