package com.valor.vod.common.web.tools;

import com.valor.vod.api.model.common.response.ExtendedResponseStatus;
import com.valor.vod.api.model.common.response.ResponseStatus;
import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.tools.http.ErrorMsgBuilder;
import com.valor.vod.common.web.constant.ErrorCode;
import com.valor.vod.common.web.constant.SystemErrorCodes;
import com.valor.vod.common.web.model.BaseResponse;
import com.valor.vod.common.web.util.TraceContext;

import common.base.tools.exception.ApiException;
import common.config.tools.config.ConfigTools3;

import lombok.extern.slf4j.Slf4j;

import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.List;
import java.util.Locale;

/**
 * 统一异常处理器
 *
 * @author Tom Tang
 * @date 2021/7/27
 * @since 3.0.0
 */
@Slf4j
@ConditionalOnMissingBean(annotation = ExceptionHandler.class)
@ControllerAdvice
public class VodExceptionHandler {

    private static final int VOD_ERROR_STATUS = HttpCode2.SERVER_ERROR;

    @ExceptionHandler(Throwable.class)
    private ResponseEntity<ResponseStatus> handThrowable(WebRequest request, Throwable e) {
        log.error("An internal exception occurred. req:{}.", request, e);
        ResponseStatus errorResponse =
                buildResponseStatus(
                        VodException.builder()
                                .language(getLanguage(request))
                                .retCode(HttpCode2.RET_SYS_EXCEPTION)
                                .errCode(HttpCode2.RET_SYS_EXCEPTION)
                                .message("Internal Server Error.")
                                .build());

        return ResponseEntity.status(HttpCode2.SERVER_ERROR).body(errorResponse);
    }

    @ExceptionHandler(IllegalArgumentException.class)
    private ResponseEntity<ResponseStatus> handIllegalArgumentException(
            WebRequest request, IllegalArgumentException e) {
        log.warn("[Bad Request]", e);
        ResponseStatus errorResponse =
                buildResponseStatus(
                        VodException.builder()
                                .language(getLanguage(request))
                                .retCode(HttpCode2.RET_INVALID_PARAM)
                                .errCode(HttpCode2.ERR_PARAM_INVALID)
                                .message(e.getMessage())
                                .build());

        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    @ExceptionHandler({VodIllegalArgumentException.class, VodIllegalStateException.class})
    private ResponseEntity<ResponseStatus> handVodIllegalException(
            WebRequest request, VodException e) {
        log.warn("[Bad Request]", e);
        return ResponseEntity.status(VOD_ERROR_STATUS).body(buildResponseStatus(e));
    }

    @ExceptionHandler(VodException.class)
    private ResponseEntity<ResponseStatus> handVodException(WebRequest request, VodException e) {
        return ResponseEntity.status(VOD_ERROR_STATUS).body(buildResponseStatus(e));
    }

    @ExceptionHandler(UndeclaredThrowableException.class)
    private ResponseEntity<ResponseStatus> handVodException(
            WebRequest request, UndeclaredThrowableException e) {
        Throwable undeclaredThrowable = e.getUndeclaredThrowable();
        if (undeclaredThrowable instanceof ApiException) {
            return handApiException(request, (ApiException) undeclaredThrowable);
        } else if (undeclaredThrowable instanceof BusinessException) {
            return handBusinessException(request, (BusinessException) undeclaredThrowable);
        } else if (undeclaredThrowable instanceof VodException) {
            return handVodException(request, (VodException) undeclaredThrowable);
        } else {
            return handThrowable(request, undeclaredThrowable);
        }
    }

    @ExceptionHandler(ApiException.class)
    private ResponseEntity<ResponseStatus> handApiException(WebRequest request, ApiException e) {
        log.warn("A api exception occurred.", e);
        ResponseStatus errorResponse =
                buildResponseStatus(
                        VodException.builder()
                                .language(getLanguage(request))
                                .retCode(e.getRetCode())
                                .errCode(e.getErrCode())
                                .message("Internal Server Error.")
                                .build());
        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    @ExceptionHandler(BusinessException.class)
    private ResponseEntity<ResponseStatus> handBusinessException(
            WebRequest request, BusinessException e) {
        log.warn("A business exception occurred.", e);
        ResponseStatus errorResponse =
                buildResponseStatus(
                        VodException.builder()
                                .language(getLanguage(request))
                                .retCode(e.getRetCode())
                                .errCode(e.getErrCode())
                                .message("Internal Server Error.")
                                .build());
        return ResponseEntity.status(VOD_ERROR_STATUS).body(errorResponse);
    }

    /**
     * 业务异常
     *
     * @param request
     * @param ex
     * @return
     */
    @ExceptionHandler(BizResponseException.class)
    public ResponseEntity<BaseResponse<Object>> handleBizResponseException(
            WebRequest request, BizResponseException ex) {
        if (ex.isLog()) {
            log.warn("A business exception occurred.", ex);
        }
        BaseResponse<Object> resp = new BaseResponse<>();
        ErrorCode errorCode = ex.getErrorCode();
        String code = ErrorManager.getRespCode(errorCode);
        String language = getLanguage(request);
        resp.setCode(code);
        resp.setMsg(ConfigTools3.getString(language + "." + code, ex.getMessage()));
        setAndRemoveTraceId(resp);
        return ResponseEntity.status(errorCode.getHttpCode()).body(resp);
    }

    /**
     * 处理参数校验错误
     *
     * @param ex
     * @return
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ResponseEntity<BaseResponse<Object>> handleMethodArgumentException(
            MethodArgumentNotValidException ex) {
        BaseResponse<Object> resp = new BaseResponse<>();
        resp.setCode(ErrorManager.getRespCode(SystemErrorCodes.PARAM_INVALID));
        List<ObjectError> errors = ex.getBindingResult().getAllErrors();

        if (errors.size() > 0) {
            ObjectError error = errors.get(0);
            if (StringUtils.hasText(error.getDefaultMessage())) {
                resp.setMsg(error.getDefaultMessage());
            } else {
                StringBuilder sb =
                        new StringBuilder(SystemErrorCodes.PARAM_INVALID.getDesc())
                                .append(" in object '")
                                .append(error.getObjectName())
                                .append("'");
                if (error instanceof FieldError) {
                    FieldError fieldError = (FieldError) error;
                    sb.append(" on field '").append(fieldError.getField()).append("'");
                }
                resp.setMsg(sb.toString());
            }
        } else {
            resp.setMsg(SystemErrorCodes.PARAM_INVALID.getDesc());
        }
        setAndRemoveTraceId(resp);

        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resp);
    }

    private void setAndRemoveTraceId(ResponseStatus errorResponse) {
        errorResponse.setTraceId(TraceContext.getTraceId());
        TraceContext.removeTraceId();
    }

    private void setAndRemoveTraceId(BaseResponse<?> errorResponse) {
        errorResponse.setTraceId(TraceContext.getTraceId());
        TraceContext.removeTraceId();
    }

    private String getLanguage(WebRequest request) {
        String language = request.getHeader(HttpHeaders.ACCEPT_LANGUAGE);
        if (!StringUtils.hasText(language)) {
            language = Locale.ENGLISH.getLanguage();
        }
        return language;
    }

    private ResponseStatus buildResponseStatus(VodException e) {
        ExtendedResponseStatus<Object> errorResponse = new ExtendedResponseStatus<>();
        String errorMessage =
                ErrorMsgBuilder.buildErrMsgNotAppendCode(
                        e.getLanguage(),
                        e.getDeviceId(),
                        e.getRetCode(),
                        e.getErrCode(),
                        e.getMessage());
        errorResponse.setStatus(e.getRetCode(), e.getErrCode(), errorMessage);
        errorResponse.setErrMessage(errorMessage);
        setAndRemoveTraceId(errorResponse);

        SystemErrorCodes errorCode = SystemErrorCodes.getByRetCode(e.getRetCode());
        errorResponse.setCode(ErrorManager.getRespCode(errorCode));
        errorResponse.setMsg(errorMessage);
        return errorResponse;
    }
}
