package com.valor.vod.common.web.tools;

import com.valor.vod.common.web.constant.ErrorCode;
import com.valor.vod.common.web.constant.SystemErrorCodes;
import com.valor.vod.common.web.constant.VodHttpAttr;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * Vod http上下文
 *
 * @author Bruce Wu
 * @since 2022-09-30
 */
public class VodHttpContext {

    private static final ThreadLocal<VodHttpContext> LOCAL =
            ThreadLocal.withInitial(VodHttpContext::new);

    private final Map<VodHttpAttr, Object> attrMap = new HashMap<>();

    public VodHttpContext() {}

    public static VodHttpContext getContext() {
        return LOCAL.get();
    }

    public static void setContext(VodHttpContext context) {
        LOCAL.set(context);
    }

    public static void removeContext() {
        LOCAL.remove();
    }

    /**
     * 设置属性
     *
     * @param key 属性key
     * @param value 属性value
     * @return
     */
    public VodHttpContext set(VodHttpAttr key, Object value) {
        this.attrMap.put(key, value);
        return this;
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @throws BizResponseException 属性value为NULL时抛异常
     * @return Object
     */
    public Object get(VodHttpAttr key) {
        return get(key, SystemErrorCodes.ATTR_NOT_FOUND);
    }

    /**
     * 获取属性value，为空返回默认值
     *
     * @param key 属性key
     * @return Object
     */
    public Object getOrDefault(VodHttpAttr key, Object defaultValue) {
        Object o = get(key, null);
        return Objects.isNull(o) ? defaultValue : o;
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @param nullErrorCode 属性value为抛出的异常错误码
     * @throws BizResponseException nullErrorCode不为NULL且属性value为NULL时抛异常
     * @return Object
     */
    public Object get(VodHttpAttr key, ErrorCode nullErrorCode) {
        Object o = this.attrMap.get(key);
        if (Objects.isNull(o) && Objects.nonNull(nullErrorCode)) {
            throw new BizResponseException(
                    nullErrorCode, nullErrorCode.getDesc() + ":[" + key.getName() + "]");
        }
        return o;
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @throws BizResponseException 属性value为NULL时抛异常
     * @return String
     */
    public String getString(VodHttpAttr key) {
        return getString(key, SystemErrorCodes.ATTR_NOT_FOUND);
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @param nullErrorCode
     * @throws BizResponseException
     * @return String
     */
    public String getString(VodHttpAttr key, ErrorCode nullErrorCode) {
        Object o = get(key, nullErrorCode);
        if (Objects.isNull(o)) {
            return null;
        }
        return o.toString();
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @throws BizResponseException 属性value为NULL时抛异常
     * @return Long
     */
    public Long getLong(VodHttpAttr key) {
        return getLong(key, SystemErrorCodes.ATTR_NOT_FOUND);
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @param nullErrorCode 属性value为抛出的异常错误码
     * @throws BizResponseException nullErrorCode不为NULL且属性value为NULL时抛异常
     * @return Long
     */
    public Long getLong(VodHttpAttr key, ErrorCode nullErrorCode) {
        Object o = get(key, nullErrorCode);
        if (Objects.isNull(o)) {
            return null;
        }
        if (o instanceof Long) {
            return (Long) o;
        }
        return Long.parseLong(o.toString());
    }

    /**
     * 获取属性value
     *
     * @param key 属性key
     * @throws BizResponseException 属性value为NULL时抛异常
     * @return Integer
     */
    public Integer getInteger(VodHttpAttr key) {
        return getInteger(key, SystemErrorCodes.ATTR_NOT_FOUND);
    }

    /**
     * 获取属性value，值为NULL时抛业务异常
     *
     * @param key 属性key
     * @param nullErrorCode 属性value为抛出的异常错误码
     * @throws BizResponseException nullErrorCode不为NULL且属性value为NULL时抛异常
     * @return Integer
     */
    public Integer getInteger(VodHttpAttr key, ErrorCode nullErrorCode) {
        Object o = get(key, nullErrorCode);
        if (Objects.isNull(o)) {
            return null;
        }
        if (o instanceof Integer) {
            return (Integer) o;
        }
        return Integer.parseInt(o.toString());
    }

    /**
     * 移除属性
     *
     * @param key 属性key
     * @return
     */
    public VodHttpContext remove(VodHttpAttr key) {
        this.attrMap.remove(key);
        return this;
    }

    /**
     * 移除全部属性
     *
     * @return
     */
    public VodHttpContext removeAll() {
        this.attrMap.clear();
        return this;
    }

    public static Map<VodHttpAttr, Object> getCopyOfContentMap() {
        return new HashMap<>(getContext().attrMap);
    }

    public static void setContentMap(Map<VodHttpAttr, Object> attrMap) {
        getContext().attrMap.putAll(attrMap);
    }
}
